<?php

namespace Drupal\api_browser\Form;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\KeyValueStore\KeyValueFactoryInterface;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\project_browser\Plugin\ProjectBrowserSourceManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Api Browser Service Form.
 */
class ApiBrowserServiceForm extends EntityForm {

  /**
   * ProjectBrowserSourceManager Service.
   *
   * @var \Drupal\project_browser\Plugin\ProjectBrowserSourceManager
   */
  protected ProjectBrowserSourceManager $projectBrowserSourceManager;

  /**
   * Key Value Factory Service.
   *
   * @var \Drupal\Core\KeyValueStore\KeyValueFactoryInterface
   */
  protected KeyValueFactoryInterface $keyValueFactory;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    $instance = parent::create($container);
    $instance->projectBrowserSourceManager = $container->get(ProjectBrowserSourceManager::class);
    $instance->keyValueFactory = $container->get('keyvalue.expirable');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state): array {
    $form = parent::form($form, $form_state);

    /** @var \Drupal\api_browser\Entity\ApiBrowserService $entity */
    $entity = $this->entity;
    $form['name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Name'),
      '#maxlength' => 255,
      '#default_value' => $entity->label(),
      '#description' => $this->t('Name for service.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->id(),
      '#machine_name' => [
        'exists' => '\Drupal\api_browser\Entity\ApiBrowserService::load',
      ],
      '#disabled' => !$entity->isNew(),
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $entity->get('description'),
      '#description' => $this->t('Details about the service and what it does.'),
    ];

    $form['categories'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Categories'),
      '#default_value' => $entity->get('categories') ?? '',
      '#description' => $this->t('List of categories to use for the projects. Each category should be on a new line in the format <code>key|label</code>.'),
    ];

    $form['listing'] = [
      '#type' => 'details',
      '#title' => $this->t('Listing Details'),
      '#tree' => TRUE,
      '#open' => TRUE,
    ];
    $form['listing']['endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Endpoint'),
      '#default_value' => $entity->get('listing')['endpoint'] ?? '',
      '#description' => $this->t('URL endpoint to use to get a list of the projects.'),
      '#required' => TRUE,
    ];
    $form['listing']['method'] = [
      '#type' => 'select',
      '#title' => $this->t('Request Method'),
      '#default_value' => $entity->get('listing')['method'] ?? '',
      '#description' => $this->t('Request method to use for searching. Currently only GET/POST are supported.'),
      '#options' => [
        'GET' => $this->t('GET'),
        'POST' => $this->t('POST'),
      ],
      '#required' => TRUE,
    ];
    $form['listing']['auth'] = [
      '#type' => 'select',
      '#title' => $this->t('Authentication Type'),
      '#default_value' => $entity->get('listing')['auth'] ?? '',
      '#description' => $this->t('Authentication method to use for the listing endpoint.'),
      '#options' => $this->getAuthTypes(),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::authTypeListingCallback',
        'wrapper' => 'listing-credentials-wrapper',
      ],
    ];

    $form['listing']['credentials'] = [
      '#type' => 'container',
      '#title' => $this->t('Credentials'),
      '#prefix' => '<div id="listing-credentials-wrapper">',
      '#suffix' => '</div>',
    ];

    $auth_type = $form_state->getValue('listing')['auth'] ?? '';
    // @todo Find a better way to handle this.
    if ($auth_type) {
      $config_screen = $this->getAuthConfigScreen($auth_type, $entity->get('listing')['credentials'] ?? []);
      if (!empty($config_screen)) {
        $form['listing']['credentials']['#type'] = 'details';
        $form['listing']['credentials']['#open'] = TRUE;
        $form['listing']['credentials'] = array_merge($form['listing']['credentials'], $config_screen);
      }
    }

    $form['listing']['path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Results Path'),
      '#default_value' => $entity->get('listing')['path'] ?? '',
      '#description' => $this->t('Path to use for looping through the listings. Following should be in @link format.', [
        '@link' => Link::fromTextAndUrl('JMESPath', Url::fromUri('https://jmespath.org/'))->toString(),
      ]),
      '#required' => TRUE,
    ];

    $form['listing']['pagination'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Listing allows for pagination'),
      '#description' => $this->t('This endpoint contains a way to page through results.'),
      '#default_value' => $entity->get('listing')['pagination'] ?? FALSE,
    ];

    $states = [
      'invisible' => [
        ':input[name="listing[pagination]"]' => ['checked' => FALSE],
      ],
      'required' => [
        ':input[name="listing[pagination]"]' => ['checked' => TRUE],
      ],
    ];

    $form['listing']['page'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Variable to use for pagination'),
      '#default_value' => $entity->get('listing')['page'] ?? '',
      '#description' => $this->t('Variable name to use for pagination throughout the listings. This will get sent in the request.'),
      '#states' => $states,
    ];

    $form['listing']['per_page'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Variable to use for per page limit'),
      '#default_value' => $entity->get('listing')['per_page'] ?? '',
      '#description' => $this->t('Variable name to use for pagination throughout the listings. This will get sent in the request.'),
      '#states' => $states,
    ];

    $form['search'] = [
      '#type' => 'details',
      '#title' => $this->t('Search Parameters'),
      '#description' => $this->t('Parameters that can be used when querying the search results.'),
      '#prefix' => '<div id="search-parameters-fieldset-wrapper">',
      '#suffix' => '</div>',
      '#open' => TRUE,
    ];

    $header = [
      'parameter' => $this->t('Parameter'),
      'value' => $this->t('Value'),
      'remove' => $this->t('Remove'),
    ];

    $form['search']['search'] = [
      '#type' => 'table',
      '#header' => $header,
      '#empty' => $this->t('No parameters added'),
      '#tree' => TRUE,
    ];

    // Get the number of names in the form already.
    $lines = $form_state->get('lines');
    // We have to ensure that there is at least one name field.
    if ($lines === NULL) {
      $searchFields = $entity->get('search') ?? [];
      $form_state->set('lines', $searchFields);
      $lines = $searchFields;
    }

    foreach ($lines as $key => $searchField) {
      $tmp = [];
      $tmp['parameter'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Parameter'),
        '#title_display' => 'invisible',
        '#default_value' => $searchField['parameter'] ?? '',
        '#required' => TRUE,
      ];
      $tmp['value'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Value'),
        '#title_display' => 'invisible',
        '#default_value' => $searchField['value'] ?? '',
      ];
      $tmp['remove'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove'),
        '#submit' => ['::removeParameter'],
        '#ajax' => [
          'callback' => '::parameterCallback',
          'wrapper' => 'search-parameters-fieldset-wrapper',
        ],
        '#name' => 'remove_' . $key,
        '#validate' => [],
        '#limit_validation_errors' => [],
      ];

      $form['search']['search'][$key] = $tmp;
    }

    $form['search']['actions']['add_more'] = [
      '#type' => 'submit',
      '#value' => $this->t('Add another search parameter'),
      '#submit' => ['::addParameter'],
      '#ajax' => [
        'callback' => '::parameterCallback',
        'wrapper' => 'search-parameters-fieldset-wrapper',
      ],
      '#limit_validation_errors' => [['search']],
      '#validate' => [],
    ];

    $form['project'] = [
      '#type' => 'details',
      '#title' => $this->t('Project Details'),
      '#tree' => TRUE,
      '#open' => TRUE,
    ];
    $form['project']['enable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use a separate endpoint to get project info.'),
      '#default_value' => $entity->get('project')['enable'] ?? FALSE,
    ];

    $states = [
      'invisible' => [
        ':input[name="project[enable]"]' => ['checked' => FALSE],
      ],
      'required' => [
        ':input[name="project[enable]"]' => ['checked' => TRUE],
      ],
    ];

    $form['project']['endpoint'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Project Endpoint'),
      '#default_value' => $entity->get('project')['endpoint'] ?? '',
      '#description' => $this->t('Endpoint to return details about a project in the list. Variables can be used. Example <code>{{ item_key }}</code> refers to the key of the item in the list.'),
      '#states' => $states,
    ];
    $form['project']['method'] = [
      '#type' => 'select',
      '#title' => $this->t('Request Method'),
      '#default_value' => $entity->get('project')['method'] ?? '',
      '#description' => $this->t('Method to use to get project details.'),
      '#options' => [
        'GET' => $this->t('GET'),
        'POST' => $this->t('POST'),
      ],
      '#required' => TRUE,
      '#states' => $states,
    ];
    $form['project']['auth'] = [
      '#type' => 'select',
      '#title' => $this->t('Authentication Type'),
      '#default_value' => $entity->get('project')['auth'] ?? '',
      '#description' => $this->t('Authentication method to use for project details.'),
      '#options' => $this->getAuthTypes(),
      '#required' => TRUE,
      '#ajax' => [
        'callback' => '::authTypeProjectCallback',
        'wrapper' => 'project-credentials-wrapper',
      ],
      '#states' => $states,
    ];

    $auth_type = $form_state->getValue('project')['auth'] ?? '';

    $form['project']['credentials'] = [
      '#type' => 'container',
      '#title' => $this->t('Credentials'),
      '#prefix' => '<div id="project-credentials-wrapper">',
      '#suffix' => '</div>',
      '#states' => $states,
    ];

    if ($auth_type) {
      $config_screen = $this->getAuthConfigScreen($auth_type, $entity->get('project')['credentials'] ?? []);
      if (!empty($config_screen)) {
        $form['project']['credentials']['#type'] = 'details';
        $form['project']['credentials']['#open'] = TRUE;
        $form['project']['credentials'] = array_merge($form['project']['credentials'], $config_screen);
      }
    }

    $form['project']['path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Results Path'),
      '#default_value' => $entity->get('project')['path'] ?? '',
      '#description' => $this->t('Path to use for getting project information. Following should be in @link format. Variables can be used. Example <code>{{ item_key }}</code> refers to the key of the item in the list.', [
        '@link' => Link::fromTextAndUrl('JMESPath', Url::fromUri('https://jmespath.org/'))->toString(),
      ]),
      '#states' => $states,
    ];

    $form['field_mapping'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Project Field Mapping'),
      '#tree' => TRUE,
    ];

    $form['field_mapping']['description'] = [
      '#type' => 'markup',
      '#markup' => implode('</p><p>',
        [
          $this->t('The following section is used to map the fields from the response to the fields required for a Project Browser Project'),
          $this->t(
            'The fields below are simple textareas that can leverage @url to transform the data and apply changes to the requested data.', [
              '@url' => Link::fromTextAndUrl('Twig', Url::fromUri('https://twig.symfony.com/'))->toString(),
            ]),
          $this->t('The variables refer to the keys that are returned for each project.'),
        ]
      ) . '</p>',
    ];

    $form['field_mapping']['type'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Project Type'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['type'] ?? '',
      '#description' => $this->t('Type of the project. Module, Recipe, Theme, etc.'),
    ];
    $form['field_mapping']['package_name'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Package Name'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['package_name'] ?? '',
      '#description' => $this->t('The name of the package to use for display.'),
    ];
    $form['field_mapping']['machine_name'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Machine Name'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['machine_name'] ?? '',
      '#description' => $this->t('Machine name of project. This will be ran through a function to make this a machine name.'),
    ];
    $form['field_mapping']['url'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('URL to Project'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['url'] ?? '',
      '#description' => $this->t('URL for more project information.'),
    ];
    $form['field_mapping']['logo'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Logo'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['logo'] ?? '',
      '#description' => $this->t('Url to use for the logo of the project.'),
    ];
    $form['field_mapping']['title'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Title'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['title'] ?? '',
      '#description' => $this->t('The title of the project.'),
    ];
    $form['field_mapping']['short_description'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Short Description'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['short_description'] ?? '',
      '#description' => $this->t('A short description for the project.'),
    ];
    $form['field_mapping']['long_description'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Long Description'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['long_description'] ?? '',
      '#description' => $this->t('A long description for the project.'),
    ];
    $form['field_mapping']['compatible'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Is Compatible?'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['compatible'] ?? '',
      '#description' => $this->t('Whether the project is compatible with the current version of Drupal. Following should return a true/false.'),
    ];
    $form['field_mapping']['maintained'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Is Maintained'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['maintained'] ?? '',
      '#description' => $this->t('Whether the project is considered to be maintained or not. Following should return a true/false.'),
    ];
    $form['field_mapping']['covered'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Is Covered'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['covered'] ?? '',
      '#description' => $this->t('Whether the project is considered to be covered or not. Following should return a true/false.'),
    ];
    $form['field_mapping']['active'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Is Active'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['active'] ?? '',
      '#description' => $this->t('Whether the project is considered to be active or not. Following should return a true/false.'),
    ];
    $form['field_mapping']['star_user_count'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Star User Count'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['star_user_count'] ?? '',
      '#description' => $this->t('User start count of the project. Should return a number.'),
    ];
    $form['field_mapping']['project_usage_total'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Project Usage Total'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['project_usage_total'] ?? '',
      '#description' => $this->t('Total usage of the project. Should return a number.'),
    ];
    $form['field_mapping']['created'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Created'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['created'] ?? '',
      '#description' => $this->t('When was the project created last timestamp. Should return a timestamp.'),
    ];
    $form['field_mapping']['changed'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Changed'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['changed'] ?? '',
      '#description' => $this->t('When was the project changed last timestamp. Should return a timestamp.'),
    ];
    $form['field_mapping']['author'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Author(s)'),
      '#required' => TRUE,
      '#default_value' => $entity->get('field_mapping')['author'] ?? '',
      '#description' => $this->t('Author of the project. Should return a comma list of the authors.'),
    ];
    $form['field_mapping']['categories'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Categories'),
      '#default_value' => $entity->get('field_mapping')['categories'] ?? '',
      '#description' => $this->t('Categories of the project. Should return a comma list of categories.'),
    ];
    $form['field_mapping']['images'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Project Images'),
      '#default_value' => $entity->get('field_mapping')['images'] ?? '',
      '#description' => $this->t('Images of the project. Should return a list of image urls in comma separated format.'),
    ];
    $form['field_mapping']['warnings'] = [
      '#type' => 'textarea',
      '#rows' => 2,
      '#title' => $this->t('Project Warnings'),
      '#default_value' => $entity->get('field_mapping')['warnings'] ?? '',
      '#description' => $this->t('Warnings for the project. Should return a comma separated list.'),
    ];

    $form['testing'] = [
      '#type' => 'container',
      '#access' => $this->config('api_browser.settings')->get('debug')['enable'] ?? FALSE,
    ];

    $form['testing']['listing'] = [
      '#type' => 'container',
      '#prefix' => '<div id="listing-test-wrapper">',
      '#suffix' => '</div>',
    ];

    $form['testing']['listing']['title'] = [
      '#markup' => '<h2>Listing Endpoint Test</h2>',
    ];

    if (($listingTestOutput = $form_state->get('listing_test')) != '') {
      $form['testing']['listing']['output'] = [
        '#type' => 'markup',
        '#markup' => $listingTestOutput,
        '#prefix' => '<div><pre>',
        '#suffix' => '</pre></div>',
      ];
    }

    $form['testing']['listing']['test_submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Test Listing Endpoint'),
      '#submit' => ['::listingTest'],
      '#ajax' => [
        'callback' => '::listingTestCallback',
        'wrapper' => 'listing-test-wrapper',
      ],
      '#limit_validation_errors' => [['listing'], ['search']],
      '#validate' => [],
    ];

    $form['testing']['project'] = [
      '#type' => 'container',
      '#prefix' => '<div id="project-test-wrapper">',
      '#suffix' => '</div>',
    ];
    $form['testing']['project']['title'] = [
      '#markup' => '<h2>Project Endpoint Test</h2>',
    ];

    if (($projectTestOutput = $form_state->get('project_test')) != '') {
      $form['testing']['project']['output'] = [
        '#type' => 'markup',
        '#markup' => $projectTestOutput,
        '#prefix' => '<div><pre>',
        '#suffix' => '</pre></div>',
      ];
    }

    $form['testing']['project']['test_submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Test Project Endpoint'),
      '#submit' => ['::projectTest'],
      '#ajax' => [
        'callback' => '::projectTestCallback',
        'wrapper' => 'project-test-wrapper',
      ],
      '#limit_validation_errors' => [['project'], ['field_mapping']],
      '#validate' => [],
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    $search = $form_state->getValue('search');
    if (is_string($search) && empty($search)) {
      $form_state->setValue('search', []);
    }
    parent::validateForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state): int {
    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $this->entity->label()];
    $message = $result == SAVED_NEW
      ? $this->t('Created new Project Browser API Browser Service %label.', $message_args)
      : $this->t('Updated Project Browser API Browser Service %label.', $message_args);
    $this->messenger()->addStatus($message);
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    $this->projectBrowserSourceManager->clearCachedDefinitions();
    Cache::invalidateTags(['api_browser:results:' . $this->entity->id()]);

    $categories = $this->keyValueFactory->get('project_browser')->get('categories');
    // @phpstan-ignore method.notFound
    $categories['api_browser_project:' . $this->entity->id()] = $this->entity->getCategories();
    $this->keyValueFactory->get('project_browser')->set('categories', $categories);

    return $result;
  }

  /**
   * Return the search array key for callback.
   *
   * @param array $form
   *   Form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form State Data.
   *
   * @return array
   *   Return the form element.
   */
  public function parameterCallback(array &$form, FormStateInterface $form_state): array {
    return $form['search'];
  }

  /**
   * Callback for adding parameter row.
   */
  public function addParameter(array &$form, FormStateInterface $form_state): void {
    $search = $form_state->getValue('search') ?? [];
    $lines = array_map(function ($item) {
      return [
        'parameter' => $item['parameter'],
        'value' => $item['value'],
      ];
    }, $search);

    $lines[] = [];
    $form_state->set('lines', $lines);
    $form_state->setRebuild();
  }

  /**
   * Callback for removing parameter row.
   */
  public function removeParameter(array &$form, FormStateInterface $form_state): void {
    $trigger = $form_state->getTriggeringElement();
    $removeKey = (int) str_replace('remove_', '', $trigger['#name']);
    $input = $form_state->getUserInput();
    $lines = $input['search'] ?? [];
    array_splice($lines, $removeKey, 1);
    $form_state->set('lines', $lines);
    $form_state->setValue('search', $lines);
    $input['search'] = $lines;
    $form_state->setUserInput($input);
    $form_state->setRebuild();
  }

  /**
   * Callback for listing endpoint test.
   */
  public function listingTest(array &$form, FormStateInterface $form_state): void {
    // @phpstan-ignore method.notFound
    $output = $this->entity->testListingEndpoint();
    $form_state->set('listing_test', $output);
    $form_state->setRebuild();
  }

  /**
   * Callback for project endpoint test.
   */
  public function projectTest(array &$form, FormStateInterface $form_state): void {
    // @phpstan-ignore method.notFound
    $output = $this->entity->testProjectEndpoint();
    $form_state->set('project_test', $output);
    $form_state->setRebuild();
  }

  /**
   * Callback for Ajax of Auth Type.
   */
  public function authTypeListingCallback(array $form, FormStateInterface $form_state): array {
    return $form['listing']['credentials'];
  }

  /**
   * Callback for Ajax of Auth Type.
   */
  public function authTypeProjectCallback(array $form, FormStateInterface $form_state): array {
    return $form['project']['credentials'];
  }

  /**
   * Callback for Ajax with Listing Test.
   */
  public function listingTestCallback(array $form, FormStateInterface $form_state): array {
    return $form['testing']['listing'];
  }

  /**
   * Callback for Ajax with Project Test.
   */
  public function projectTestCallback(array $form, FormStateInterface $form_state): array {
    return $form['testing']['project'];
  }

  /**
   * Return the Config Screen for the provided auth type.
   */
  protected function getAuthConfigScreen(string $auth_type, array $credentials = []): array {
    $config_screen = [];
    switch ($auth_type) {
      case 'api_key':
        $config_screen['key'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Key'),
          '#default_value' => $credentials['key'] ?? '',
          '#required' => TRUE,
        ];
        $config_screen['value'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Value'),
          '#default_value' => $credentials['value'] ?? '',
          '#required' => TRUE,
        ];
        $config_screen['location'] = [
          '#type' => 'select',
          '#options' => [
            'header' => $this->t('Header'),
            'query' => $this->t('Query Parameter'),
          ],
          '#title' => $this->t('Key Location'),
          '#default_value' => $credentials['location'] ?? '',
          '#required' => TRUE,
        ];
        break;

      case 'bearer_token':
        $config_screen['token'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Token Value'),
          '#default_value' => $credentials['token'] ?? '',
          '#required' => TRUE,
        ];
        break;

      case 'basic_auth':
        $config_screen['username'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Username'),
          '#default_value' => $credentials['username'] ?? '',
          '#required' => TRUE,
        ];
        $config_screen['password'] = [
          '#type' => 'textfield',
          '#title' => $this->t('Password'),
          '#default_value' => $credentials['password'] ?? '',
          '#required' => TRUE,
        ];
        break;
    }

    $this->moduleHandler->alter('api_browser_auth_config_screen', $config_screen, $auth_type, $credentials);

    return $config_screen;
  }

  /**
   * Return the Authentication Types.
   */
  protected function getAuthTypes(): array {
    $list = [
      'no_auth' => $this->t('No Auth'),
      'api_key' => $this->t('API Key'),
      'bearer_token' => $this->t('Bearer Token'),
      'basic_auth' => $this->t('Basic Auth'),
    ];

    $this->moduleHandler->alter('api_browser_auth_types', $list);

    return $list;
  }

}
