<?php

namespace Drupal\api_browser\Plugin\ProjectBrowserSource;

use Drupal\api_browser\Entity\ApiBrowserService;
use Drupal\project_browser\Plugin\ProjectBrowserSourceBase;
use Drupal\project_browser\ProjectBrowser\Project;
use Drupal\project_browser\ProjectBrowser\ProjectsResultsPage;
use Drupal\project_browser\SecurityStatus;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * API Project Browser Source Plugin.
 *
 * @ProjectBrowserSource(
 *   id = "api_browser_project",
 *   label = @Translation("API Project Browser Source"),
 *   description = @Translation("Create Project Browser Sources from API Endpoints"),
 *   deriver = "\Drupal\api_browser\Plugin\Derivative\ApiBrowserProjectBrowser"
 * )
 */
class ApiBrowserProjectBrowserSource extends ProjectBrowserSourceBase {

  /**
   * Service for Plugin.
   *
   * @var \Drupal\api_browser\Entity\ApiBrowserService|null
   */
  protected ?ApiBrowserService $source = NULL;

  /**
   * Constructors a new ApiBrowserProjectBrowserSource object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Psr\Log\LoggerInterface $logger
   *   A logger instance.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, protected LoggerInterface $logger) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->source = $plugin_definition['service'] instanceof ApiBrowserService ? $plugin_definition['service'] : NULL;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    // @phpstan-ignore-next-line
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('logger.factory')->get('api_project_browser'),
    );
  }

  /**
   * {@inheritDoc}
   */
  public function getProjects(array $query = []): ProjectsResultsPage {
    $projects = $this->source?->getList($query) ?? [];

    // Filter by project machine name.
    if (!empty($query['machine_name'])) {
      $projects = array_filter($projects, fn(Project $project) => $project->machineName === $query['machine_name']);
    }

    // Filter by coverage.
    if (!empty($query['security_advisory_coverage']) && $query['security_advisory_coverage'] === SecurityStatus::Covered->value) {
      $projects = array_filter($projects, fn(Project $project) => $project->isCovered);
    }

    // Filter by categories.
    if (!empty($query['categories'])) {
      // @phpstan-ignore-next-line
      $projects = array_filter($projects, fn(Project $project) => array_intersect(array_column($project->categories, 'id'), explode(',', $query['categories'])));
    }

    // Filter by search text.
    if (!empty($query['search'])) {
      $projects = array_filter($projects, fn(Project $project) => stripos($project->title, $query['search']) !== FALSE);
    }

    // Filter by sorting criterion.
    if (!empty($query['sort'])) {
      $sort = $query['sort'];
      switch ($sort) {
        case 'a_z':
          usort($projects, fn($x, $y) => $x->title <=> $y->title);
          break;

        case 'z_a':
          usort($projects, fn($x, $y) => $y->title <=> $x->title);
          break;
      }
    }
    $project_count = count($projects);
    if (!empty($query['page']) && !empty($query['limit'])) {
      $projects = array_chunk($projects, $query['limit'])[$query['page']] ?? [];
    }

    return $this->createResultsPage($projects, $project_count);
  }

  /**
   * {@inheritDoc}
   */
  public function getCategories(): array {
    return $this->source?->getCategories() ?? [];
  }

}
