<?php

namespace Drupal\api_data_connector\Controller;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Controller\ControllerBase;
use GuzzleHttp\ClientInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for API Data Connector.
 */
class ApiDataConnectorController extends ControllerBase {

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The HTTP client service.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * ApiDataConnectorController constructor.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ClientInterface $http_client,
    LoggerInterface $logger,
  ) {
    $this->configFactory = $config_factory;
    $this->httpClient = $http_client;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('http_client'),
      $container->get('logger.factory')->get('api_data_connector')
    );
  }

  /**
   * Provides autocomplete suggestions for a specific field.
   *
   * This function handles incoming requests for autocomplete suggestions
   * based on the user's input. It retrieves relevant data from the database
   * or an external API and returns a JSON response with the suggestions.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The HTTP request object containing the user's input.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   A JSON response containing an array of suggestions.
   */
  public function autocomplete(Request $request) {
    $input = $request->query->get('q');
    $api_url = $request->query->get('api_url');
    $query_parameter = $request->query->get('query');
    $query_key = $request->query->get('key');
    $query_value = $request->query->get('value');
    // Query the user entities based on the input.
    if ($input && $api_url) {
      $options = [];
      $queryParams = [
        $query_parameter => $input,
      ];
      try {
        // Make a GET request to the API to fetch user data.
        $response = $this->httpClient->request('GET', $api_url, [
          'query' => $queryParams,
        ]);
        $data = json_decode($response->getBody()->getContents(), TRUE);
        // Assuming the API returns an array of users with 'id' and 'name'.
        foreach ($data as $user) {
          if (isset($user[$query_key]) && isset($user[$query_value])) {
            $options[] = [
              'value' => $user[$query_value] . '(' . $user[$query_key] . '),',
              'label' => $user[$query_value],
            ];
          }
        }
      }
      catch (\Exception $e) {
        // Log an error message if the API call fails.
        $this->logger->error('Failed to fetch user options from API: @message', ['@message' => $e->getMessage()]);
      }
    }

    return new JsonResponse($options);
  }

}
