<?php

namespace Drupal\api_data_connector\Plugin\Field\FieldType;

use Drupal\Core\Field\Attribute\FieldType;
use Drupal\Core\Field\FieldItemBase;
use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TypedData\DataDefinition;

/**
 * Plugin implementation of the 'api_data_connector_item' field type.
 */
#[FieldType(
  id: "api_data_connector_item",
  label: new TranslatableMarkup("API Data Connector Field."),
  description: new TranslatableMarkup("API Data Connector field."),
  default_widget: "api_data_connector_widget",
  default_formatter: "api_data_connector_formatter",
)]
class ApiDataConnectorItem extends FieldItemBase {

  /**
   * {@inheritdoc}
   */
  public static function defaultFieldSettings() {
    return [
      'api_fieldset' => [
        'api_url' => '',
        'query_parameter' => '',
      ],
      'mapping_fieldset' => [
        'key' => '',
        'value' => '',
      ],
      'auth_fieldset' => [
        'auth_value' => '',
      ],
    ] + parent::defaultFieldSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function fieldSettingsForm(array $form, FormStateInterface $form_state) {
    $element = [];
    $element['api_fieldset'] = [
      '#type' => 'details',
      '#title' => $this->t('API Details'),
    ];
    $element['mapping_fieldset'] = [
      '#type' => 'details',
      '#title' => $this->t('Mapping Details'),
    ];
    $element['auth_fieldset'] = [
      '#type' => 'details',
      '#title' => $this->t('Authentication Details'),
    ];
    $element['api_fieldset']['api_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('API URL'),
      '#default_value' => ($this->getSetting('api_fieldset') != NULL) ? $this->getSetting('api_fieldset')['api_url'] : '',
      '#required' => TRUE,
      '#description' => $this->t('Field to entering the API URL for API based autocomplete.'),
    ];
    $element['api_fieldset']['query_parameter'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Parameter Name'),
      '#required' => TRUE,
      '#default_value' => ($this->getSetting('api_fieldset') != NULL) ? $this->getSetting('api_fieldset')['query_parameter'] : '',
      '#description' => $this->t('Field for entering the query parameter name for API based autocomplete.'),
    ];
    $element['mapping_fieldset']['key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Mapping Key'),
      '#default_value' => ($this->getSetting('mapping_fieldset') != NULL) ? $this->getSetting('mapping_fieldset')['key'] : '',
      '#required' => TRUE,
      '#description' => $this->t('Specify the key to map from the API response, such as: id, userid, user_id, or node_id.'),
    ];
    $element['mapping_fieldset']['value'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Mapping Value'),
      '#default_value' => ($this->getSetting('mapping_fieldset') != NULL) ? $this->getSetting('mapping_fieldset')['value'] : '',
      '#required' => TRUE,
      '#description' => $this->t('Specify the value to map from the API response, such as: name, username, user_name, or node.'),
    ];
    $element['auth_fieldset']['auth_value'] = [
      '#type' => 'select',
      '#title' => $this->t('Authentication Method'),
      '#default_value' => 'none',
      '#required' => TRUE,
      '#description' => $this->t('Field for selecting an authentication method.'),
      '#options' => [
        'none' => $this->t('None'),
      ],
    ];
    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public static function propertyDefinitions(FieldStorageDefinitionInterface $field_definition) {
    $properties = [];

    // Define your properties here.
    $properties['target_id'] = DataDefinition::create('integer')
      ->setLabel(t('Target ID'))
      ->setRequired(TRUE);
    $properties['target_name'] = DataDefinition::create('string')
      ->setLabel(t('Target Name'))
      ->setRequired(TRUE);
    // Add other properties as needed.
    return $properties;
  }

  /**
   * {@inheritdoc}
   */
  public static function schema(FieldStorageDefinitionInterface $field_definition) {
    return [
      'columns' => [
        'target_id' => [
          'type' => 'int',
          'not null' => FALSE,
        ],
        'target_name' => [
          'type' => 'varchar',
          'length' => 255,
          'not null' => TRUE,
        ],
      ],
    ];
  }

}
