<?php

namespace Drupal\api_data_connector\Plugin\Field\FieldWidget;

use Drupal\Core\Field\Attribute\FieldWidget;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\Plugin\Field\FieldWidget\OptionsWidgetBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use GuzzleHttp\ClientInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Plugin implementation of the 'api_data_connector_select' widget.
 */
#[FieldWidget(
  id: 'api_data_connector_select',
  label: new TranslatableMarkup('Api Data Connector Widget Options Select2'),
  field_types: [
    'api_data_connector_item',
  ],
  multiple_values: TRUE,
)]
class ApiDataConnectorOptionsSelectWidget extends OptionsWidgetBase {

  /**
   * The HTTP client service.
   *
   * @var \GuzzleHttp\Client
   */
  protected $httpClient;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * ApiDataConnectorController constructor.
   *
   * @param \GuzzleHttp\ClientInterface $http_client
   *   The HTTP client service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(
    $plugin_id,
    $plugin_definition,
    FieldDefinitionInterface $field_definition,
    array $settings,
    array $third_party_settings,
    ClientInterface $http_client,
    LoggerInterface $logger,
  ) {
    parent::__construct(
      $plugin_id,
      $plugin_definition,
      $field_definition,
      $settings,
      $third_party_settings
    );
    $this->httpClient = $http_client;
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(
    ContainerInterface $container,
    array $configuration,
    $plugin_id,
    $plugin_definition
  ) {
    return new static(
      $plugin_id,
      $plugin_definition,
      $configuration['field_definition'],
      $configuration['settings'],
      $configuration['third_party_settings'],
      $container->get('http_client'),
      $container->get('logger.factory')->get('api_data_connector')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function formElement(FieldItemListInterface $items, $delta, array $element, array &$form, FormStateInterface $form_state) {
    $element = parent::formElement($items, $delta, $element, $form, $form_state);
    $api_url = ($this->getFieldSetting('api_fieldset') != NULL) ? $this->getFieldSetting('api_fieldset')['api_url'] : '';
    $key = ($this->getFieldSetting('mapping_fieldset') != NULL) ? $this->getFieldSetting('mapping_fieldset')['key'] : '';
    $value = ($this->getFieldSetting('mapping_fieldset') != NULL) ? $this->getFieldSetting('mapping_fieldset')['value'] : '';
    $options = $this->fetchSelectOptions($api_url, $key, $value);
    $target_names = array_column($items->getValue(), 'target_name');
    $element += [
      '#type' => 'select2',
      '#options' => $options,
      // Do not display a 'multiple' select box if there is only one option.
      '#multiple' => $this->multiple && count($options) > 1,
      '#default_value' => !empty($target_names) ? $target_names : NULL,
    ];

    return $element;
  }

  /**
   * {@inheritdoc}
   */
  public function massageFormValues(array $values, array $form, FormStateInterface $form_state) {
    foreach ($values as $key => $value) {
      // The entity_autocomplete form element returns an array when an entity
      // was "autocreated", so we need to move it up a level.
      preg_match_all('/\((\d+)\)/', $value['target_id'], $ids);
      // Step 2: Build the desired array structure.
      $elements = explode(',', rtrim($value['target_id'], ', '));
      $target_names = array_map('trim', $elements);
      unset($values[$key]['target_id']);
      foreach ($ids[1] as $key_id => $id) {
        $values['target_id'][$key] = [
          'target_id' => trim($id),
          'target_name' => $target_names[$key_id],
        ];
      }
    }
    return $values['target_id'];
  }

  /**
   * Fetch user options from the API.
   *
   * @param string $api_url
   *   The URL of the API endpoint to fetch user data from.
   * @param string $key
   *   The key in the user data array that represents the id.
   * @param string $value
   *   The key in the user data array that represents the value.
   *
   * @return array
   *   An associative array of select option id and values.
   */
  private function fetchSelectOptions($api_url, $key, $value) {
    $options = [];
    try {
      // Make a GET request to the API to fetch user data.
      $response = $this->httpClient->request('GET', $api_url);
      $data = json_decode($response->getBody()->getContents(), TRUE);
      // Assuming the API returns an array of users with 'userid' and 'username'.
      foreach ($data as $user) {
        $user_key = $this->fetchKey($user, $key);
        $user_value = $this->fetchKey($user, $value);
        // Ensure that 'userid' and 'username' exist in the user data.
        if (isset($user_key) && isset($user_value)) {
          $options[$user_value . '(' . $user_key . ')'] = $user_value;
        }
      }
    } catch (\Exception $e) {
      $this->messenger()
        ->addError($this->t('The API for fetching the user group is not functioning properly. </br> @message', [
          '@message' => $e->getMessage(),
        ]));
      // Log an error message if the API call fails.
      $this->logger->error('Failed to fetch user options from API: @message', ['@message' => $e->getMessage()]);
    }

    return $options;
  }

  /**
   * Fetch user options from the API.
   *
   * @param string $user
   *   Array result.
   * @param string $key
   *   The key in the user data array that represents the id or value.
   *
   * @return array
   *   A string contain the id or value.
   */
  private function fetchKey($user, $key) {
    $key_explode = explode('.', $key);

    // Build the nested array structure
    $current = $user;
    foreach ($key_explode as $key_value) {
      $current = $current[$key_value];
    }
    return $current;
  }
}
