<?php

namespace Drupal\api_proxy\Controller;

use Drupal\api_proxy\Plugin\HttpApiInterface;
use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheableResponse;
use Drupal\Core\Http\Exception\CacheableBadRequestHttpException;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Main controller to forward requests.
 */
final class Forwarder {

  const QUERY_PARAM_URI = '_api_proxy_uri';

  /**
   * Forwards incoming requests to the connected API.
   *
   * @param \Drupal\api_proxy\Plugin\HttpApiInterface $api_proxy
   *   The API proxy plugin.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The incoming request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The response object.
   */
  public function forward(HttpApiInterface $api_proxy, Request $request): Response {
    $third_party_uri = $this->sanitizeUri($request->query->get(static::QUERY_PARAM_URI));
    $cache_contexts = ['url.query_args:' . static::QUERY_PARAM_URI];
    $cacheability = (new CacheableMetadata())->addCacheContexts($cache_contexts);
    if (empty($third_party_uri)) {
      throw new CacheableBadRequestHttpException(
        $cacheability,
        sprintf('Unable to find a valid URI in the %s query parameter.', static::QUERY_PARAM_URI)
      );
    }
    $response = $api_proxy->forward($request, $third_party_uri);
    if ($response instanceof CacheableResponse) {
      $response->addCacheableDependency($cacheability);
    }
    return $response;
  }

  private function sanitizeUri(string $uri) {
    return UrlHelper::isValid($uri) ? $uri : '';
  }

}
