<?php

namespace Drupal\api_proxy\EventSubscriber;

use Drupal\api_proxy\Controller\Forwarder;
use Drupal\api_proxy\Plugin\HttpApiPluginBase;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheableResponse;
use Symfony\Cmf\Component\Routing\RouteProviderInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\GetResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Handles options requests.
 */
class OptionsRequestSubscriber implements EventSubscriberInterface {

  const ROUTE_NAME = 'api_proxy.forwarder';

  /**
   * The route provider.
   *
   * @var \Symfony\Cmf\Component\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * The decorated service.
   *
   * @var \Symfony\Component\EventDispatcher\EventSubscriberInterface
   */
  protected $subject;

  /**
   * Creates a new OptionsRequestSubscriber instance.
   *
   * @param \Symfony\Cmf\Component\Routing\RouteProviderInterface $route_provider
   *   The route provider.
   * @param \Symfony\Component\EventDispatcher\EventSubscriberInterface
   *   The decorated service.
   */
  public function __construct(RouteProviderInterface $route_provider, EventSubscriberInterface $subject) {
    $this->routeProvider = $route_provider;
    $this->subject = $subject;
  }

  /**
   * Tries to handle the options request.
   *
   * @param \Symfony\Component\HttpKernel\Event\GetResponseEvent $event
   *   The request event.
   */
  public function onRequest(GetResponseEvent $event) {
    $request = $event->getRequest();
    $routes = $this->routeProvider->getRouteCollectionForRequest($event->getRequest());
    if ($request->getMethod() !== 'OPTIONS') {
      return;
    }
    $route_name = current(array_filter(
      array_keys($routes->all()),
      function ($route_name) {
        return $route_name === static::ROUTE_NAME;
      }
    ));
    if (!$route_name) {
      $this->subject->onRequest($event);
      return;
    }
    $param_name = key($routes->get($route_name)->getOption('parameters'));
    $proxy = $request->attributes->get($param_name);
    if (!$proxy instanceof HttpApiPluginBase) {
      $cacheability = (new CacheableMetadata())->addCacheTags(['route']);
      $response = new CacheableResponse('', 404);
      $response->addCacheableDependency($cacheability);
      $event->setResponse($response);
      return;
    }
    $response = $proxy->corsResponse($request);
    $cache_contexts = [
      'url.query_args:' . Forwarder::QUERY_PARAM_URI,
      'headers:Origin',
    ];
    $cacheability = (new CacheableMetadata())
      ->addCacheContexts($cache_contexts);
    $response->addCacheableDependency($cacheability);
    $response->setVary('Origin', FALSE);
    $event->setResponse($response);
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    // Set a high priority so it is executed before routing.
    $events[KernelEvents::REQUEST][] = ['onRequest', 31];
    return $events;
  }

}
