<?php

namespace Drupal\api_toolkit\EventSubscriber;

use Drupal\api_toolkit\Exception\ApiValidationException;
use Drupal\api_toolkit\Validation\ContextFactory;
use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Cache\CacheableJsonResponse;
use Drupal\Core\EventSubscriber\HttpExceptionSubscriberBase;
use Drupal\Core\EventSubscriber\MainContentViewSubscriber;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;
use Symfony\Component\HttpKernel\Exception\HttpExceptionInterface;
use Symfony\Component\Validator\ConstraintViolationListInterface;

/**
 * Handles json error responses with validation results in a standardised way.
 */
class ExceptionJsonSubscriber extends HttpExceptionSubscriberBase {

  use StringTranslationTrait;

  /**
   * The context factory.
   *
   * @var \Drupal\api_toolkit\Validation\ContextFactory
   */
  protected $context;

  /**
   * Constructs a new ExceptionJsonSubscriber.
   *
   * @param \Drupal\api_toolkit\Validation\ContextFactory $context
   *   A context factory.
   */
  public function __construct(ContextFactory $context) {
    $this->context = $context;
  }

  /**
   * {@inheritdoc}
   */
  protected function getHandledFormats() {
    return ['json'];
  }

  /**
   * {@inheritdoc}
   */
  protected static function getPriority() {
    return -40;
  }

  /**
   * {@inheritdoc}
   */
  public function onException(ExceptionEvent $event) {
    $request = $event->getRequest();
    if (!$this->isJsonRequest($request)) {
      return;
    }

    $exception = $event->getThrowable();
    if ($exception instanceof ApiValidationException) {
      $violations = $exception->getViolations();

      if ($violations === NULL) {
        $context = $this->context->create();
        if ($exception->getMessage() !== NULL) {
          $context->addViolation($exception->getMessage());
        }
        $violations = $context->getViolations();
      }

      $response = $this->createJsonResponse($exception, $violations);
      $event->setResponse($response);

      return;
    }

    $context = $this->context->create();
    $context->addViolation($this->getMessage($exception));
    $response = $this->createJsonResponse($exception, $context->getViolations());
    $event->setResponse($response);
  }

  /**
   * Build a general error message in case no validation errors are present.
   */
  protected function getMessage(\Throwable $exception): string {
    if (error_displayable()) {
      return $exception->getMessage();
    }

    if ($exception instanceof AccessDeniedHttpException) {
      return $this->t('Access denied.');
    }

    return $this->t('The website encountered an unexpected error. Please try again later.');
  }

  /**
   * Build a standardised JSON response based on a list of validation errors.
   */
  protected function createJsonResponse(\Throwable $exception, ConstraintViolationListInterface $violations): JsonResponse {
    $status = Response::HTTP_BAD_REQUEST;
    $headers = [];
    $data = [];

    /** @var \Symfony\Component\Validator\ConstraintViolationInterface $violation */
    foreach ($violations as $violation) {
      if ($path = $violation->getPropertyPath()) {
        $data['errors'][] = [
          'path' => $path,
          'message' => $violation->getMessage(),
        ];
      }
      else {
        $data['errors'][] = [
          'message' => $violation->getMessage(),
        ];
      }
    }

    if ($exception instanceof HttpExceptionInterface) {
      $status = $exception->getStatusCode();
      $headers = $exception->getHeaders();
    }

    if (!isset($headers['Content-Type'])) {
      $headers['Content-Type'] = 'application/json';
    }

    // If the exception is cacheable, generate a cacheable response.
    if ($exception instanceof CacheableDependencyInterface) {
      $response = new CacheableJsonResponse($data, $status, $headers);
      $response->addCacheableDependency($exception);

      return $response;
    }

    return new JsonResponse($data, $status, $headers);
  }

  /**
   * Determines whether a request accepts a JSON response.
   */
  protected function isJsonRequest(Request $request): bool {
    $format = $request->query->get(MainContentViewSubscriber::WRAPPER_FORMAT, $request->getRequestFormat());
    if ($format === 'json') {
      return TRUE;
    }

    if (in_array('application/json', $request->getAcceptableContentTypes(), TRUE)) {
      return TRUE;
    }

    return FALSE;
  }

}
