<?php

namespace Drupal\api_toolkit\Plugin\Validation\Constraint;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Validates the EntityExists constraint.
 */
class EntityExistsValidator extends ConstraintValidator implements ContainerInjectionInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->entityTypeManager = $container->get('entity_type.manager');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function validate($uuid, Constraint $constraint): void {
    if (!$constraint instanceof EntityExists) {
      throw new UnexpectedTypeException($constraint, EntityExists::class);
    }

    if ($uuid === NULL) {
      return;
    }

    try {
      $entityType = $this->entityTypeManager->getDefinition($constraint->entityTypeId);
      $storage = $this->entityTypeManager->getStorage($constraint->entityTypeId);
    }
    catch (InvalidPluginDefinitionException | PluginNotFoundException $e) {
      $this->context->addViolation($constraint->invalidEntityType, [
        '%value' => $constraint->entityTypeId,
      ]);
      return;
    }

    $entity = $storage->loadByProperties(['uuid' => $uuid]);
    $entity = reset($entity);

    if (!$entity instanceof EntityInterface || ($constraint->bundle && !$entity->bundle() === $constraint->bundle)) {
      $this->context->addViolation($constraint->invalidId, [
        '@entityType' => $this->getEntityTypeLabel($entityType, $constraint->bundle),
        '%value' => $uuid,
      ]);
    }
  }

  /**
   * Get the singular label of an entity type or bundle.
   */
  protected function getEntityTypeLabel(EntityTypeInterface $entityType, ?string $bundle): string {
    if ($bundleEntityType = $entityType->getBundleEntityType()) {
      $bundleDefinition = $this->entityTypeManager
        ->getStorage($bundleEntityType)
        ->load($bundle);

      if ($bundleDefinition) {
        return strtolower($bundleDefinition->label());
      }
    }

    return $entityType->getSingularLabel();
  }

}
