<?php

namespace Drupal\api_toolkit\Plugin\Validation\Constraint;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\migrate\MigrateException;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\ConstraintValidator;
use Symfony\Component\Validator\Exception\UnexpectedTypeException;

/**
 * Validates the MigrationSourceExists constraint.
 */
class MigrationSourceExistsValidator extends ConstraintValidator implements ContainerInjectionInterface {

  /**
   * The migration plugin manager.
   *
   * @var \Drupal\migrate\Plugin\MigrationPluginManagerInterface
   */
  protected $migrationManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = new static();
    $instance->migrationManager = $container->get('plugin.manager.migration');

    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function validate($id, Constraint $constraint): void {
    if (!$constraint instanceof MigrationSourceExists) {
      throw new UnexpectedTypeException($constraint, MigrationSourceExists::class);
    }

    if ($id === NULL) {
      return;
    }

    /** @var \Drupal\migrate\Plugin\MigrationInterface|false $migration */
    $migration = $this->migrationManager->createInstance($constraint->migration);
    if ($migration === FALSE) {
      $this->context->addViolation('The "%id" migration does not exist.', ['%id' => $constraint->migration]);
      return;
    }

    $idMap = $migration->getIdMap();

    try {
      $records = $idMap->lookupDestinationIds([$id]);
      if ($records === []) {
        $this->context->addViolation($constraint->invalidId, [
          '%value' => $id,
          '%migration' => $constraint->migration,
        ]);
      }
    }
    catch (MigrateException $e) {
      $this->context->addViolation($e->getMessage());
      return;
    }
  }

}
