<?php

namespace Drupal\api_toolkit\Request;

use Drupal\api_toolkit\Exception\ApiValidationException;
use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Cache\CacheableDependencyTrait;
use Symfony\Component\HttpFoundation\Response;
use function GuzzleHttp\json_decode;
use Symfony\Component\HttpFoundation\Request;

/**
 * Base class for API requests.
 */
abstract class ApiRequestBase implements ApiRequestInterface, CacheableDependencyInterface {

  use CacheableDependencyTrait;

  /**
   * Get an array with all properties of the given object.
   *
   * We use get_object_vars because uninitialized (not provided) properties are
   * missing from this array. This way, we can differentiate between
   * uninitialized and null values.
   */
  public function all(): array {
    return get_object_vars($this);
  }

  /**
   * Checks whether a property exists and is initialized.
   */
  public function has(string $key): bool {
    return array_key_exists($key, $this->all());
  }

  /**
   * {@inheritdoc}
   */
  public static function fromRequest(Request $request): self {
    // Collect values
    $values = array_merge(
      $request->attributes->all(),
      $request->request->all(),
      $request->query->all(),
    );

    if ($request->getContentType() === 'json') {
      try {
        $content = json_decode($request->getContent(), TRUE, 512, JSON_THROW_ON_ERROR);
        $values = array_merge($values, $content);
      }
      catch (\JsonException $exception) {
        throw ApiValidationException::create(
          NULL,
          Response::HTTP_BAD_REQUEST,
          sprintf('Error while parsing json: %s', $exception->getMessage()),
          $exception
        );
      }
    }

    // Create instance
    $instance = static::fromValues($values);

    // Add cacheability metadata
    $reflection = new \ReflectionClass($instance);
    $cacheContexts = [];
    $cacheTags = [];

    foreach ($reflection->getProperties() as $property) {
      if ($property->getDeclaringClass()->getName() === self::class) {
        continue;
      }

      $attribute = $request->attributes->get($property->getName());
      if ($attribute instanceof CacheableDependencyInterface) {
        $cacheContexts = array_merge($cacheContexts, $attribute->getCacheContexts());
        $cacheTags = array_merge($cacheTags, $attribute->getCacheTags());
      }

      $cacheContexts[] = 'url.query_args:' . $property->getName();
    }

    $instance->cacheContexts = $cacheContexts;
    $instance->cacheTags = $cacheTags;

    return $instance;
  }

  /**
   * Create a new instance from an array of values.
   */
  public static function fromValues(array $values): self {
    $instance = new static();
    $reflection = new \ReflectionClass($instance);

    foreach ($reflection->getProperties() as $property) {
      $key = $property->getName();

      if (array_key_exists($key, $values)) {
        $instance->{$property->getName()} = $values[$key];
      }
    }

    return $instance;
  }

}
