<?php

namespace Drupal\api_toolkit\EventSubscriber;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\EventSubscriber\MainContentViewSubscriber;
use Drupal\Core\Routing\RouteMatch;
use Drupal\Core\Site\MaintenanceModeInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Maintenance mode subscriber to return a JSON response.
 */
class MaintenanceModeSubscriber implements EventSubscriberInterface {

  /**
   * The maintenance mode.
   *
   * @var \Drupal\Core\Site\MaintenanceMode
   */
  protected $maintenanceMode;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $config;

  /**
   * Constructs a new MaintenanceModeSubscriber.
   *
   * @param \Drupal\Core\Site\MaintenanceModeInterface $maintenance_mode
   *   The maintenance mode.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(MaintenanceModeInterface $maintenance_mode, ConfigFactoryInterface $config_factory) {
    $this->maintenanceMode = $maintenance_mode;
    $this->config = $config_factory;
  }

  /**
   * Return an error response if site is in maintenance mode.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The event to process.
   */
  public function onKernelRequestMaintenance(RequestEvent $event) {
    $request = $event->getRequest();
    if (!$this->isJsonRequest($request)) {
      return;
    }

    $route_match = RouteMatch::createFromRequest($request);

    if ($this->maintenanceMode->applies($route_match)) {
      $response = new JsonResponse([
        'errors' => [
              ['message' => $this->getSiteMaintenanceMessage()],
        ],
      ], Response::HTTP_SERVICE_UNAVAILABLE);
      $event->setResponse($response);
    }
  }

  /**
   * Gets the site maintenance message.
   *
   * @return \Drupal\Component\Render\MarkupInterface
   *   The formatted site maintenance message.
   */
  protected function getSiteMaintenanceMessage() {
    return new FormattableMarkup($this->config->get('system.maintenance')->get('message'), [
      '@site' => $this->config->get('system.site')->get('name'),
    ]);
  }

  /**
   * Determines whether a request accepts a JSON response.
   */
  protected function isJsonRequest(Request $request): bool {
    $format = $request->query->get(MainContentViewSubscriber::WRAPPER_FORMAT, $request->getRequestFormat());
    if ($format === 'json') {
      return TRUE;
    }

    if (in_array('application/json', $request->getAcceptableContentTypes(), TRUE)) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events[KernelEvents::REQUEST][] = ['onKernelRequestMaintenance', 35];
    return $events;
  }

}
