<?php

namespace Drupal\api_toolkit\Normalizer;

use Drupal\api_toolkit\Exception\ApiValidationException;
use Drupal\api_toolkit\Request\ApiRequestBase;
use Drupal\api_toolkit\Request\ApiRequestInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\Constraints\Choice;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Type;
use Symfony\Component\Validator\ConstraintViolation;
use Symfony\Component\Validator\ConstraintViolationList;
use Symfony\Component\Validator\ConstraintViolationListInterface;
use function GuzzleHttp\json_decode;

/**
 *
 */
class ApiRequestNormalizer extends ObjectNormalizer {

  /**
   * A string map with values that should always be mapped to other values.
   *
   * @var string[]
   */
  const VALUE_MAP = [
    '' => NULL,
    'null' => NULL,
    'true' => TRUE,
    'false' => FALSE,
    '[]' => NULL,
  ];

  /**
   * A string map from possible type declarations to types returned by gettype().
   *
   * @var string[]
   */
  const TYPE_MAP = [
    'integer' => 'int',
    'double' => 'float',
    'boolean' => 'bool',
    'array' => 'array',
    'string' => 'string',
  ];

  /**
   * {@inheritdoc}
   */
  public function supportsNormalization($data, $format = null): bool
  {
      return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function denormalize($data, $type, $format = NULL, array $context = []) {
    /** @var \Symfony\Component\HttpFoundation\Request $data */

    // Collect all values.
    $values = array_merge(
      $data->request->all(),
      $data->query->all(),
    );

    // Add non-internal attributes.
    foreach ($data->attributes->all() as $key => $value) {
      if (strpos($key, '_') === 0) {
        continue;
      }

      $values[$key] = $value;
    }

    if ($data->getContentType() === 'json' && $content = $data->getContent()) {
      try {
        $content = json_decode($content, TRUE, 512, JSON_THROW_ON_ERROR);
        $values = array_merge($values, $content);
      }
      catch (\JsonException $exception) {
        throw ApiValidationException::create(
          NULL,
          Response::HTTP_BAD_REQUEST,
          sprintf('Error while parsing json: %s', $exception->getMessage()),
          $exception
        );
      }
    }

    // Collect all request-specific properties.
    // Skip properties from the base class.
    $reflection = new \ReflectionClass($type);
    $properties = array_filter(
        $reflection->getProperties(),
        function ($property) {
          return $property->getDeclaringClass()->getName() !== ApiRequestBase::class;
        }
    );

    // Transform values.
    $originalValues = $values;

    foreach ($properties as $property) {
      $name = $property->getName();

      if (!isset($values[$name])) {
        continue;
      }

      // Transform values to their proper scalar types.
      $value = &$values[$name];

      if (is_string($value) && array_key_exists($value, self::VALUE_MAP)) {
        $value = self::VALUE_MAP[$value];
      }

      if (is_array($value)) {
        foreach ($value as $subKey => $subValue) {
          if (is_string($subValue) && array_key_exists($subValue, self::VALUE_MAP)) {
            $value[$subKey] = $subValue = self::VALUE_MAP[$subValue];
          }
          if (is_null($subValue)) {
            unset($value[$subKey]);
          }
        }
      }

      $propertyType = $property->getType();

      if ($propertyType instanceof \ReflectionNamedType) {
        if (enum_exists($propertyType->getName()) && method_exists($propertyType->getName(), 'from')) {
          if ($enum = $propertyType->getName()::tryFrom($value)) {
            $value = $enum;
          }
        }

        if ($propertyType->getName() === 'array' && $value === NULL) {
          $value = [];
        }
      }

      if (($propertyType instanceof \ReflectionUnionType || ($this->isPhp81() && $type instanceof \ReflectionIntersectionType))) {
        foreach ($propertyType->getTypes() as $singleType) {
          if ($value === NULL && $singleType === 'array') {
            $value = [];
          }
        }
      }

      unset($value);
    }

    // Check whether the values are allowed to be assigned to their properties.
    // @todo Use DenormalizerInterface::COLLECT_DENORMALIZATION_ERRORS once Drupal supports Symfony 5.4
    // @see https://symfony.com/blog/new-in-symfony-5-4-serializer-improvements#collect-denormalization-type-errors
    $violations = new ConstraintViolationList();

    foreach ($properties as $property) {
      $name = $property->getName();
      $propertyType = $property->getType();

      if (!isset($values[$name])) {
        if (!$propertyType->allowsNull() && !$property->hasDefaultValue()) {
          $constraint = new NotBlank();
          $this->addViolation($violations, $constraint->message, $name, $originalValues[$name] ?? NULL, NULL, $constraint);
        }

        continue;
      }

      $value = $values[$name];
      $valueType = self::TYPE_MAP[gettype($value)] ?? gettype($value);

      if ($propertyType instanceof \ReflectionNamedType) {
        if ($this->isPhp81() && enum_exists($propertyType->getName()) && !$value instanceof \BackedEnum) {
          $options = array_column($propertyType->getName()::cases(), 'value');
          $message = sprintf("The value '%s' is not a valid option. Choose one of the following: %s", $value, implode(', ', $options));
          $this->addViolation($violations, $message, $name, $originalValues[$name] ?? NULL);
        }

        if (in_array($propertyType->getName(), self::TYPE_MAP) && $propertyType->getName() !== $valueType) {
          $this->addTypeViolation($propertyType, $property->getName(), $originalValues[$name], $value, $violations);
        }
      }

      if ($propertyType instanceof \ReflectionUnionType) {
        $hasMatch = FALSE;
        foreach ($propertyType->getTypes() as $singleType) {
          if (in_array($singleType->getName(), self::TYPE_MAP) && $singleType->getName() === $valueType) {
            $hasMatch = TRUE;
            break;
          }
        }
        if (!$hasMatch) {
          $this->addTypeViolation($propertyType->getTypes()[0], $property->getName(), $originalValues[$name], $value, $violations);
        }
      }
    }

    if ($violations->count() > 0) {
      throw ApiValidationException::create($violations);
    }

    // Add cacheability metadata.
    foreach ($properties as $property) {
      $name = $property->getName();

      // Every property might come from a query param, so add the right cache contexts.
      $values['cacheContexts'][] = 'url.query_args:' . $name;
    }

    return parent::denormalize($values, $type, $format, $context);
  }

  /**
   * {@inheritdoc}
   */
  public function supportsDenormalization($data, $type, $format = NULL) {
    return $data instanceof Request
      && is_a($type, ApiRequestInterface::class, TRUE);
  }

  protected function addTypeViolation(\ReflectionType $propertyType, string $name, $originalValue, $value, ConstraintViolationListInterface $violations) {
    $typeName = $propertyType->getName();
    $constraint = new Type([
      'type' => self::TYPE_MAP[$typeName] ?? $typeName,
    ]);
    $message = str_replace('{{ type }}', $typeName, $constraint->message);

    $this->addViolation($violations, $message, $name, $value, $originalValue, $constraint);
  }

  protected function addViolation(ConstraintViolationListInterface $violations, string $message, string $name, $value, $originalValue = NULL, ?Constraint $constraint = NULL) {
    $violation = new ConstraintViolation(
      $message,
      NULL,
      [],
      $originalValue ?? $value,
      $name,
      $value,
      NULL,
      NULL,
      $constraint
    );

    $violations->add($violation);
  }

  /**
   * Check whether the current PHP version is 8.1 or higher.
   */
  protected function isPhp81(): bool
  {
    return version_compare(phpversion(), '8.1.0', '>=');
  }

}
