<?php

namespace Drupal\api_toolkit\Normalizer;

use Drupal\api_toolkit\Exception\ApiValidationException;
use Drupal\api_toolkit\Plugin\Validation\Constraint\Enum;
use Drupal\api_toolkit\Request\ApiRequestBase;
use Drupal\api_toolkit\Request\ApiRequestInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\PropertyAccess\PropertyAccessorInterface;
use Symfony\Component\PropertyInfo\PropertyTypeExtractorInterface;
use Symfony\Component\Serializer\Mapping\ClassDiscriminatorResolverInterface;
use Symfony\Component\Serializer\Mapping\Factory\ClassMetadataFactoryInterface;
use Symfony\Component\Serializer\NameConverter\NameConverterInterface;
use Symfony\Component\Serializer\Normalizer\ObjectNormalizer;
use Symfony\Component\Validator\Constraint;
use Symfony\Component\Validator\Constraints\NotNull;
use Symfony\Component\Validator\Constraints\Type;
use Symfony\Component\Validator\ConstraintViolation;
use Symfony\Component\Validator\ConstraintViolationList;
use Symfony\Component\Validator\ConstraintViolationListInterface;
use Symfony\Component\Validator\Validator\ValidatorInterface;
use function GuzzleHttp\json_decode;

/**
 * Denormalizes classes implementing ApiRequestInterface.
 *
 * @see \Drupal\api_toolkit\Request\ApiRequestInterface
 * @see \Drupal\api_toolkit\ArgumentResolver\ApiRequestResolver
 */
class ApiRequestNormalizer extends ObjectNormalizer {

  /**
   * A string map with values that should always be mapped to other values.
   *
   * @var string[]
   */
  const VALUE_MAP = [
    '' => NULL,
    'null' => NULL,
    'true' => TRUE,
    'false' => FALSE,
    '[]' => NULL,
  ];

  /**
   * A map from possible type declarations to types returned by gettype().
   *
   * @var string[]
   */
  const TYPE_MAP = [
    'integer' => 'int',
    'double' => 'float',
    'boolean' => 'bool',
    'array' => 'array',
    'string' => 'string',
  ];

  /**
   * The validator.
   *
   * @var \Symfony\Component\Validator\Validator\ValidatorInterface
   */
  protected $validator;

  /**
   * {@inheritdoc}
   *
   * @param \Symfony\Component\Validator\Validator\ValidatorInterface $validator
   *   The validator.
   * @param \Symfony\Component\Serializer\Mapping\Factory\ClassMetadataFactoryInterface $classMetadataFactory
   *   The class metadata factory.
   * @param \Symfony\Component\Serializer\NameConverter\NameConverterInterface $nameConverter
   *   The name converter.
   * @param \Symfony\Component\PropertyAccess\PropertyAccessorInterface $propertyAccessor
   *   The property accessor.
   * @param \Symfony\Component\PropertyInfo\PropertyTypeExtractorInterface $propertyTypeExtractor
   *   The property type extractor.
   * @param \Symfony\Component\Serializer\Mapping\ClassDiscriminatorResolverInterface $classDiscriminatorResolver
   *   The class discriminator resolver.
   * @param callable $objectClassResolver
   *   The object class resolver.
   * @param array $defaultContext
   *   The default context.
   */
  public function __construct(
    ValidatorInterface $validator,
    ClassMetadataFactoryInterface $classMetadataFactory = NULL,
    NameConverterInterface $nameConverter = NULL,
    PropertyAccessorInterface $propertyAccessor = NULL,
    PropertyTypeExtractorInterface $propertyTypeExtractor = NULL,
    ClassDiscriminatorResolverInterface $classDiscriminatorResolver = NULL,
    callable $objectClassResolver = NULL,
    array $defaultContext = []
  ) {
    $this->validator = $validator;
    parent::__construct($classMetadataFactory, $nameConverter, $propertyAccessor, $propertyTypeExtractor, $classDiscriminatorResolver, $objectClassResolver, $defaultContext);
  }

  /**
   * {@inheritdoc}
   */
  public function supportsNormalization($data, $format = NULL): bool {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function denormalize($data, $type, $format = NULL, array $context = []) {
    // Collect all values.
    if ($data instanceof Request) {
      $values = $this->getValues($data);
    }
    else {
      $values = $data;
    }

    // Collect all request-specific properties.
    // Skip properties from the base class.
    $reflection = new \ReflectionClass($type);
    $properties = array_filter(
        $reflection->getProperties(),
        function ($property) {
          return $property->getDeclaringClass()->getName() !== ApiRequestBase::class;
        }
    );

    // Transform values.
    $originalValues = $values;

    foreach ($properties as $property) {
      $name = $property->getName();

      if (!isset($values[$name])) {
        continue;
      }

      // Transform values to their proper scalar types.
      $value = &$values[$name];

      if (is_string($value) && array_key_exists($value, self::VALUE_MAP)) {
        $value = self::VALUE_MAP[$value];
      }

      if (is_array($value)) {
        foreach ($value as $subKey => $subValue) {
          if (is_string($subValue) && array_key_exists($subValue, self::VALUE_MAP)) {
            $value[$subKey] = self::VALUE_MAP[$subValue];
          }
        }
      }

      $propertyType = $property->getType();

      if ($propertyType instanceof \ReflectionNamedType) {
        if ($this->isPhp81() && enum_exists($propertyType->getName()) && method_exists($propertyType->getName(), 'from')) {
          if ($enum = $propertyType->getName()::tryFrom($value)) {
            $value = $enum;
          }
        }

        if ($propertyType->getName() === 'array' && $value === NULL) {
          $value = [];
        }

        if ($propertyType->getName() === 'string' && !$propertyType->allowsNull() && $value === NULL) {
          $value = '';
        }

        if ($propertyType->getName() === 'int' && is_numeric($value)) {
          $value = (int) $value;
        }

        if ($propertyType->getName() === 'float' && is_numeric($value)) {
          $value = (float) $value;
        }
      }

      if (($this->isPhp80() && $propertyType instanceof \ReflectionUnionType) || ($this->isPhp81() && $type instanceof \ReflectionIntersectionType)) {
        foreach ($propertyType->getTypes() as $singleType) {
          if ($value === NULL && $singleType === 'array') {
            $value = [];
          }
        }
      }

      unset($value);
    }

    // Check whether the values are allowed to be assigned to their properties.
    // @todo Use DenormalizerInterface::COLLECT_DENORMALIZATION_ERRORS once Drupal supports Symfony 5.4
    // @see https://symfony.com/blog/new-in-symfony-5-4-serializer-improvements#collect-denormalization-type-errors
    $violations = new ConstraintViolationList();

    foreach ($properties as $property) {
      $name = $property->getName();
      $propertyType = $property->getType();

      if (!$propertyType->allowsNull() && (!$this->isPhp80() || !$property->hasDefaultValue())) {
        $notNullViolations = $this->validator->validate($values[$name] ?? NULL, [new NotNull()]);

        foreach ($notNullViolations as $notNullViolation) {
          $this->addViolation($violations, NULL, $name, NULL, NULL, NULL, $notNullViolation);
        }

        if ($notNullViolations->count() > 0) {
          continue;
        }
      }

      $value = $values[$name] ?? NULL;
      $valueType = self::TYPE_MAP[gettype($value)] ?? gettype($value);

      if ($propertyType instanceof \ReflectionNamedType) {
        if ($this->isPhp81() && enum_exists($propertyType->getName()) && !$value instanceof \BackedEnum) {
          $enumViolations = $this->validator->validate($value, [new Enum(['enum' => $propertyType->getName()])]);
          foreach ($enumViolations as $enumViolation) {
            $this->addViolation($violations, NULL, $name, NULL, NULL, NULL, $enumViolation);
          }
        }

        if (in_array($propertyType->getName(), self::TYPE_MAP, TRUE) && $propertyType->getName() !== $valueType && $valueType !== 'NULL') {
          $this->addTypeViolation($propertyType, $property->getName(), $originalValues[$name], $value, $violations);
        }
      }

      if ($this->isPhp80() && $propertyType instanceof \ReflectionUnionType) {
        $hasMatch = FALSE;
        foreach ($propertyType->getTypes() as $singleType) {
          if (in_array($singleType->getName(), self::TYPE_MAP, TRUE) && $singleType->getName() === $valueType) {
            $hasMatch = TRUE;
            break;
          }
        }
        if (!$hasMatch) {
          $this->addTypeViolation($propertyType->getTypes()[0], $property->getName(), $originalValues[$name], $value, $violations);
        }
      }
    }

    if ($violations->count() > 0) {
      throw ApiValidationException::create($violations);
    }

    // Add cacheability metadata.
    foreach ($properties as $property) {
      $name = $property->getName();

      // Every property might come from a query param,
      // so add the right cache contexts.
      $values['cacheContexts'][] = 'url.query_args:' . $name;
    }

    return parent::denormalize($values, $type, $format, $context);
  }

  /**
   * {@inheritdoc}
   */
  public function supportsDenormalization($data, $type, $format = NULL): bool {
    return (is_array($data) || $data instanceof Request)
      && is_a($type, ApiRequestInterface::class, TRUE);
  }

  /**
   * Extract values from the request.
   */
  protected function getValues(Request $request): array {
    $values = array_merge(
      $request->request->all(),
      $request->query->all(),
    );

    // Add non-internal attributes.
    foreach ($request->attributes->all() as $key => $value) {
      if (strpos($key, '_') === 0) {
        continue;
      }

      $values[$key] = $value;
    }

    if ($request->getContentType() === 'json' && $content = $request->getContent()) {
      try {
        $content = json_decode($content, TRUE, 512, JSON_THROW_ON_ERROR);
        $values = array_merge($values, $content);
      }
      catch (\JsonException $exception) {
        throw ApiValidationException::create(
          NULL,
          Response::HTTP_BAD_REQUEST,
          sprintf('Error while parsing json: %s', $exception->getMessage()),
          $exception
        );
      }
    }

    return $values;
  }

  /**
   * Create a new type violation and add it to the list.
   */
  protected function addTypeViolation(\ReflectionType $propertyType, string $name, $originalValue, $value, ConstraintViolationListInterface $violations): void {
    $typeName = $propertyType->getName();
    $constraint = new Type([
      'type' => self::TYPE_MAP[$typeName] ?? $typeName,
    ]);
    $message = str_replace('{{ type }}', $typeName, $constraint->message);

    $this->addViolation($violations, $message, $name, $value, $originalValue, $constraint);
  }

  /**
   * Create a new violation and add it to the list.
   */
  protected function addViolation(
    ConstraintViolationListInterface $violations,
    ?string $message = NULL,
    ?string $name = NULL,
    $value = NULL,
    $originalValue = NULL,
    ?Constraint $constraint = NULL,
    ?ConstraintViolation $violation = NULL
  ): void {
    if ($violation instanceof ConstraintViolation) {
      $violation = new ConstraintViolation(
        $violation->getMessage() ?? $message,
        $violation->getMessageTemplate(),
        $violation->getParameters(),
        $violation->getRoot() ?? $originalValue ?? $value,
        $violation->getPropertyPath() ?: $name,
        $violation->getInvalidValue() ?? $value,
        $violation->getPlural(),
        $violation->getCode(),
        $constraint ?? $violation->getConstraint(),
      );
    }
    else {
      $violation = new ConstraintViolation(
        $message,
        NULL,
        [],
        $originalValue ?? $value,
        $name,
        $value,
        NULL,
        NULL,
        $constraint
      );
    }

    $violations->add($violation);
  }

  /**
   * Check whether the current PHP version is 8.0 or higher.
   */
  protected function isPhp80(): bool {
    return version_compare(phpversion(), '8.0.0', '>=');
  }

  /**
   * Check whether the current PHP version is 8.1 or higher.
   */
  protected function isPhp81(): bool {
    return version_compare(phpversion(), '8.1.0', '>=');
  }

}
