<?php

namespace Drupal\apitools_examples\Plugin\ApiTools\Client;

use Drupal\apitools\Api\Client\ClientBase;
use Drupal\apitools\ClientManagerInterface;
use Drupal\apitools\ClientResourceManagerInterface;
use Drupal\Component\Datetime\Time;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Serializer\Encoder\XmlEncoder;
use Symfony\Component\Serializer\Serializer;

/**
 * Creates a new ApiTools api client.
 *
 * @ApiToolsClient(
 *   id = "acalog",
 *   admin_label = @Translation("Acalog API"),
 *   admin_description = @Translation("Configure settings for the Acalog XML API"),
 *   api = "acalog",
 *   config = {
 *     "api_key" = {
 *       "title" = @Translation("API Key"),
 *       "type" = "key_select"
 *     },
 *     "catalog_id" = @Translation("Catalog ID"),
 *     "base_uri" = @Translation("Base URI")
 *   }
 * )
 */
class Acalog extends ClientBase {

  /**
   * @var String
   */

  protected $apiKey;

  protected $catalogId;

  /**
   * {@inheritdoc}
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition,
                              ClientManagerInterface $client_manager,
                              ClientResourceManagerInterface $resource_manager,
                              ConfigFactoryInterface $config_factory,
                              KeyRepositoryInterface $key_repository,
                              Time $time,
                              Serializer $serializer) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $client_manager, $resource_manager, $config_factory, $key_repository, $time);
    $this->serializer = $serializer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.apitools_client'),
      $container->get('plugin.manager.apitools_client_resource'),
      $container->get('config.factory'),
      $container->get('key.repository'),
      $container->get('datetime.time'),
      $container->get('serializer')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function init(array $options = []) {
    parent::init($options);
    $this->options->add([
      'base_uri' => $this->getConfigValue('base_uri'),
      'base_path' => 'v1',
    ]);

    $this->apiKey = $this->getConfigValue('api_key');
    $this->catalogId = $this->getConfigValue('catalog_id');
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  protected function auth() {
    $this->params->set('key', $this->apiKey);
    $this->params->set('catalog', $this->catalogId);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function request($method, $path, $options = []) {
    $this->params->add($options['query'] ?? []);
    $options['query'] = $this->params->all();
    return parent::request($method, $path, $options);
  }

  /**
   * {@inheritdoc}
   */
  protected function postRequest($response) {
    if (!empty($response)) {
      $options = [
        XmlEncoder::LOAD_OPTIONS => \LIBXML_XINCLUDE | \LIBXML_HTML_NOIMPLIED | \LIBXML_NOBLANKS | \LIBXML_PARSEHUGE | \LIBXML_NOEMPTYTAG,
      ];
      return $this->serializer->decode($response, 'xml', $options);
    }
    return parent::postRequest($response);
  }
}

