<?php

namespace Drupal\apitools_examples\Plugin\ApiTools;

use Drupal\apitools\Api\Client\ClientBase;
use Drupal\Component\Serialization\Json;

/**
 * Creates a new Auth0 Management API client.
 *
 * @ApiToolsClient(
 *   id = "auth0",
 *   admin_label = @Translation("Auth0"),
 *   api = "auth0",
 *   config = {
 *     "client_id" = {
 *       "type" = "key_select",
 *       "title" = @Translation("Client ID"),
 *     },
 *     "client_secret" = {
 *       "type" = "key_select",
 *       "title" = @Translation("Client Secret"),
 *     },
 *     "audience" = @Translation("Audience"),
 *     "base_path" = @Translation("Base Path"),
 *     "grant_type" = @Translation("Grant Type"),
 *   }
 * )
 */
class Auth0 extends ClientBase  {

  /**
   * {@inheritdoc}
   */
  protected function auth() {
    if (!$this->getToken('access_token')) {
      $options = [
        'form_params' => [
          'client_id'     => $this->getConfigValue('client_id'),
          'client_secret' => $this->getConfigValue('client_secret'),
          'grant_type'    => $this->getConfigValue('grant_type'),
          'audience'      => $this->getConfigValue('audience'),
        ],
      ];
      $response = $this->request('post', 'oauth/token', $options);
      if (empty($response['access_token'])) {
        throw new \Exception('Error retrieving access token');
      }
      if (empty($response['token_type']) || $response['token_type'] !== 'Bearer') {
        throw new \Exception('Invalid token type');
      }
      $this->setTokenExpiresIn('access_token', $response['expires_in']);
      $this->setToken('access_token', $response['access_token']);
    }

    $this->options->set('headers',[
      'authorization' => 'Bearer ' . $this->getToken('access_token'),
    ]);

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  protected function postRequest($response) {
    if ($response === NULL) {
      $this->clearToken('access_token');
      throw new \Exception('Request failed');
    }
    return Json::decode($response);
  }

  public function getUser($user_id) {
    return $this->get('api/v2/users/' . urlencode($user_id));
  }
}

