<?php

namespace Drupal\apitools_examples\Plugin\ApiTools;

use Drupal\apitools\Api\Client\ClientBase;
use Drupal\apitools\ClientManagerInterface;
use Drupal\apitools\ClientResourceManagerInterface;
use Drupal\Component\Datetime\Time;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\State\StateInterface;
use Drupal\key\KeyRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * @ApiToolsClient(
 *   id = "localist",
 *   admin_label = @Translation("Localist Client"),
 *   api = "localist",
 * )
 */
class Localist extends ClientBase {

  protected StateInterface $state;

  protected ImmutableConfig $config;

  public function __construct(array $configuration, $plugin_id, $plugin_definition,
                              ClientManagerInterface $client_manager,
                              ClientResourceManagerInterface $resource_manager,
                              ConfigFactoryInterface $config_factory,
                              KeyRepositoryInterface $key_repository,
                              Time $time,
                              SerializerInterface $serializer,
                              StateInterface $state) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $client_manager, $resource_manager, $config_factory, $key_repository, $time);
    $this->serializer = $serializer;
    $this->state = $state;
    $this->config = $config_factory->get('apitools_examples.settings')->get('localist');
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('plugin.manager.apitools_client'),
      $container->get('plugin.manager.apitools_client_resource'),
      $container->get('config.factory'),
      $container->get('key.repository'),
      $container->get('datetime.time'),
      $container->get('serializer'),
      $container->get('state')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function auth() {
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function init(array $options = []) {
    if (empty($this->config) || empty($this->config['base_uri']) || empty($this->config['base_path'])) {
      throw new \Exception('Missing base uri and/or base path');
    }
    if (!$token = $this->state->get('localist_api_access_token')) {
      throw new \Exception('Missing access token for the Localist API');
    }
    $this->options->add([
      'headers' => [
        'Authorization' => 'Bearer ' . $token,
      ],
    ]);

    $this->options->add([
      'base_uri' => $this->config['base_uri'],
      'base_path' => $this->config['base_path'],
    ]);

    return parent::init($options);
  }

  /**
   * {@inheritdoc}
   */
  protected function postRequest($response) {
    $response = parent::postRequest($response);
    $decode = $this->options->get('decode');
    if ($decode === FALSE) {
      return $response;
    }
    return $this->serializer->decode($response, 'json');
  }
}

