<?php

namespace Drupal\apitools_examples\Plugin\ApiTools;

use Drupal\apitools\Api\Client\ClientBase;
use Drupal\Component\Serialization\Json;

/**
 * Creates a new Sharepoint API client.
 *
 * @ApiToolsClient(
 *   id = "sharepoint",
 *   admin_label = @Translation("Sharepoint"),
 *   api = "sharepoint",
 *   config = {
 *     "client_id" = {
 *       "type" = "key_select",
 *       "title" = @Translation("Client ID"),
 *     },
 *     "client_secret" = {
 *       "type" = "key_select",
 *       "title" = @Translation("Client Secret"),
 *     },
 *     "site_name" = @Translation("Site Name"),
 *     "resource" = @Translation("Resource"),
 *     "grant_type" = @Translation("Grant Type"),
 *     "base_path" = @Translation("Base Path"),
 *   },
 *   client_default_controller = "\Drupal\apitools_examples\Plugin\ApiTools\DefaultController"
 * )
 */
class SharepointClient extends ClientBase {

  /**
   * {@inheritdoc}
   */
  protected function auth() {
    $this->ensureToken();

    $this->options->set('headers',[
      'authorization' => 'Bearer ' . $this->getToken('access_token'),
    ]);

    return $this;
  }

  private function ensureToken() {
    if (!$this->getToken('access_token')) {
      $this->options->add(['headers' => [
        'Content-Type' => 'application/x-www-form-urlencoded',
      ]]);
      $options = [
        'base_path' => 'https://login.microsoftonline.com',
        'form_params' => [
          'client_id'     => $this->getConfigValue('client_id'),
          'client_secret' => $this->getConfigValue('client_secret'),
          'grant_type'    => $this->getConfigValue('grant_type'),
          'resource'      => $this->getConfigValue('resource'),
        ],
      ];

      $response = $this->request('get', '[sharepoint unique path]/oauth2/token', $options);
      if (empty($response['access_token'])) {
        throw new \Exception('Error retrieving access token');
      }
      if (empty($response['token_type']) || $response['token_type'] !== 'Bearer') {
        throw new \Exception('Invalid token type');
      }
      $this->setTokenExpiresIn('access_token', $response['ext_expires_in']);
      $this->setToken('access_token', $response['access_token']);
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function postRequest($response) {
    if ($response === NULL) {
      $this->clearToken('access_token');
      throw new \Exception('Request failed');
    }
    return Json::decode($response);
  }

  public function getSiteId() {
    if (!$site_id = $this->getToken('site_id')) {
      $site_name = $this->getSiteName();
      $response = $this->get('/[sharepoint domain]:/sites/' . $site_name);
      if (empty($response['id'])) {
        throw new \Exception('Error retrieving site id');
      }
      $this->setToken('site_id', $response['id']);
    }
    return $site_id;
  }

  public function getSiteName() {
    return $this->getConfigValue('site_name');
  }
}