<?php

namespace Drupal\apitools;

use Drupal\Component\Utility\NestedArray;
use GuzzleHttp\Psr7\Response;
use Psr\Http\Message\ResponseInterface;
use Traversable;

/**
 * A hybrid class to allow array iterating and ResponseInterface functionality.
 */
class IterableResponse extends Response implements \IteratorAggregate, \Countable, \ArrayAccess {

  protected $responseData;

  protected $iterableData;

  /**
   * {@inheritdoc}
   */
  public function getIterator(): Traversable {
    return new \ArrayIterator($this->iterableData);
  }

  /**
   * {@inheritdoc}
   */
  public function count(): int {
    return count($this->iterableData);
  }

  /**
   * {@inheritdoc}
   */
  public function offsetExists(mixed $offset): bool {
    return isset($this->iterableData[$offset]);
  }

  /**
   * {@inheritdoc}
   */
  public function offsetGet(mixed $offset): mixed {
    return $this->offsetExists($offset) ? $this->iterableData[$offset] : NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function offsetSet(mixed $offset, mixed $value): void {
    $this->iterableData[$offset] = $value;
  }

  /**
   * {@inheritdoc}
   */
  public function offsetUnset(mixed $offset): void {
    unset($this->iterableData[$offset]);
  }

  /**
   * {@inheritdoc}
   */
  public function __construct(array $parents, $data, $status = 200, array $headers = [], $body = null, $version = '1.1', $reason = null) {
    parent::__construct($status, $headers, $body, $version, $reason);
    $this->responseData = $data;
    $this->iterableData = &NestedArray::getValue($data, $parents);
  }

  public static function create(array $parents, array $data, ResponseInterface $response) {
    return new static(
      $parents,
      $data,
      $response->getStatusCode(),
      $response->getHeaders(),
      $response->getBody(),
      $response->getProtocolVersion(),
      $response->getReasonPhrase()
    );
  }

  public function getData() {
    return $this->iterableData;
  }

  public function getResponseData() {
    return $this->responseData;
  }
}