<?php

namespace Drupal\apptiles\ApplicationTiles;

use Drupal\Core\Url;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Implementation of default application tiles manager.
 */
class Manager implements ManagerInterface {

  /**
   * Name of theme.
   *
   * @var string
   */
  private $themeName = '';
  /**
   * Path to theme.
   *
   * @var string
   */
  private $themePath = '';
  /**
   * Application tiles settings.
   *
   * @var array
   */
  private $settings = [];
  /**
   * Config factory.
   *
   * @var ConfigFactoryInterface
   */
  private $configFactory;
  /**
   * Theme handler.
   *
   * @var ThemeHandlerInterface
   */
  private $themeHandler;
  /**
   * Router context.
   *
   * @var AdminContext
   */
  private $adminContext;

  /**
   * ApplicationTiles constructor.
   *
   * @param ConfigFactoryInterface $config_factory
   *   Config factory.
   * @param ThemeHandlerInterface $theme_handler
   *   Theme handler.
   * @param AdminContext $admin_context
   *   Router context.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ThemeHandlerInterface $theme_handler,
    AdminContext $admin_context
  ) {
    $this->configFactory = $config_factory;
    $this->themeHandler = $theme_handler;
    $this->adminContext = $admin_context;

    $this->themeName = $theme_handler->getDefault();
    $this->themePath = drupal_get_path('theme', $this->themeName);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('theme_handler'),
      $container->get('router.admin_context')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getSettings() {
    if (empty($this->settings)) {
      $theme = $this->themeHandler->getTheme($this->themeName);
      // Use global config because "theme_get_setting()" could possibly
      // override it. Execution of the next construction will break the
      // result:
      // @code
      // \Drupal::configFactory()
      //   ->getEditable('bartik.settings')
      //   ->set(APPTILES_MODULE_NAME, [])
      //   ->save(TRUE);
      // @endcode
      $this->settings = (array) $this->configFactory->get('system.theme.global')->get(APPTILES_MODULE_NAME);

      if (!empty($theme->info[APPTILES_MODULE_NAME]) && is_array($theme->info[APPTILES_MODULE_NAME])) {
        $this->settings = array_merge($this->settings, $theme->info[APPTILES_MODULE_NAME]);
      }
    }

    return $this->settings;
  }

  /**
   * {@inheritdoc}
   */
  public function getSetting($setting, $default_value = NULL) {
    // Initialize, probably, uninitialized settings.
    $this->getSettings();

    return isset($this->settings[$setting]) ? $this->settings[$setting] : $default_value;
  }

  /**
   * {@inheritdoc}
   */
  public function getPath() {
    return "$this->themePath/tiles";
  }

  /**
   * {@inheritdoc}
   */
  public function getTiles() {
    $path = $this->getPath();
    $settings = [];

    foreach ($this->variations() as $os => $dimensions) {
      foreach ($dimensions as $dimension) {
        $settings[$os][$dimension] = "$path/$os/$dimension." . APPTILES_IMAGE_EXTENSION;
      }
    }

    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function getUrls() {
    $settings = [];

    foreach ($this->getTiles() as $os => $dimensions) {
      $settings[$os] = [];

      foreach (array_filter($dimensions, 'file_exists') as $dimension => $file) {
        $settings[$os][$dimension] = Url::fromUri("internal:/$file", ['absolute' => TRUE])->toString(TRUE)->getGeneratedUrl();
      }
    }

    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function isAvailable() {
    return !empty($this->getSetting('allowed_for_admin_theme')) || !$this->adminContext->isAdminRoute();
  }

  /**
   * {@inheritdoc}
   */
  public function variations() {
    return [
      'android' => [
        '96x96',
        '144x144',
        '192x192',
        '194x194',
      ],
      'windows' => [
        '70x70',
        '150x150',
        '310x150',
        '310x310',
      ],
      'ios' => [
        '57x57',
        '60x60',
        '72x72',
        '76x76',
        '114x114',
        '120x120',
        '144x144',
        '152x152',
        '180x180',
      ],
    ];
  }

}
