<?php

namespace Drupal\apptiles;

use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\File\Exception\NotRegularDirectoryException;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Routing\AdminContext;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;

/**
 * Implementation of the default application tiles manager.
 */
class AppTilesManager implements AppTilesManagerInterface {

  /**
   * A path to the default theme.
   *
   * @var string
   */
  protected $themePath;

  /**
   * A set of settings.
   *
   * @var array
   */
  protected $settings = [];

  /**
   * An instance of the "cache.default" service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected $cacheBackend;

  /**
   * An instance of the "router.admin_context" service.
   *
   * @var \Drupal\Core\Routing\AdminContext
   */
  protected $adminContext;

  /**
   * An instance of the "file_system" service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * {@inheritdoc}
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    ThemeHandlerInterface $theme_handler,
    CacheBackendInterface $cache_backend,
    AdminContext $admin_context,
    FileSystemInterface $file_system
  ) {
    $theme = $theme_handler->getTheme($theme_handler->getDefault());

    // Use global config because "theme_get_setting()" could possibly
    // override it. Execution of the next construction will break the
    // result:
    // @code
    // \Drupal::configFactory()
    //   ->getEditable('bartik.settings')
    //   ->set(APPTILES_MODULE_NAME, [])
    //   ->save(TRUE);
    // @endcode
    $this->settings = (array) $config_factory
      ->get('system.theme.global')
      ->get(APPTILES_MODULE_NAME);

    if (!empty($theme->info[APPTILES_MODULE_NAME]) && \is_array($theme->info[APPTILES_MODULE_NAME])) {
      $this->settings = \array_merge($this->settings, $theme->info[APPTILES_MODULE_NAME]);
    }

    $this->cacheBackend = $cache_backend;
    $this->adminContext = $admin_context;
    $this->fileSystem = $file_system;
    $this->themePath = $theme->getPath();
  }

  /**
   * {@inheritdoc}
   */
  public function getSettings(): array {
    return $this->settings;
  }

  /**
   * {@inheritdoc}
   */
  public function getSetting(string $setting, $default_value = NULL) {
    return $this->settings[$setting] ?? $default_value;
  }

  /**
   * {@inheritdoc}
   */
  public function getPath(): string {
    return "$this->themePath/tiles";
  }

  /**
   * {@inheritdoc}
   */
  public function getUrls(): array {
    $path = $this->getPath();
    $cache = $this->cacheBackend->get($path);

    if ($cache === FALSE) {
      $settings = [];

      foreach (static::TYPES as $os) {
        try {
          foreach ($this->fileSystem->scanDirectory("$path/$os", '/^\d+x\d+\.' . APPTILES_IMAGE_EXTENSION . '$/', ['recurse' => FALSE]) as $file) {
            $settings[$os][$file->name] = $file->uri;
          }
        }
        catch (NotRegularDirectoryException $e) {
        }
      }

      $this->cacheBackend->set($path, $settings);
    }
    else {
      $settings = $cache->data;
    }

    foreach (static::TYPES as $os) {
      $settings += [$os => []];
    }

    return $settings;
  }

  /**
   * {@inheritdoc}
   */
  public function isAvailable(): bool {
    return !empty($this->getSetting('allowed_for_admin_theme')) || !$this->adminContext->isAdminRoute();
  }

}
