<?php

/**
 * @file
 * Content type definitions and entity customization callbacks.
 *
 * This file contains helper functions for adding fields to different entities.
 * It is responsible for fielding the archibald_lomch_description content type,
 * the archibald_vcard and archibald_partner custom entities, as well as add
 * some fields to the core user entities.
 *
 * It also contains a lot of meta information about the different
 * archibald_lomch_description fields, like their "path" inside a JSON object,
 * and even advanced help, which is leveraged by the archibald_help module to
 * help editors better understand what certain fields actually mean.
 */

/**
 * Field definitions for our custom archibald_lomch_description content type.
 *
 * We split this into a separate function so we can store our field definitions
 * in one place.
 *
 * @see archibald_lomch_description_node_type_insert()
 *
 * @return array
 *    An array of field definitions, so that
 *    archibald_lomch_description_node_type_insert() can understand them.
 */
function archibald_archibald_lomch_description_fields() {
  $lifecycle_contribute_help_label = t("Lifecycle contributors", array(), array('context' => 'archibald:help'));
  $lifecycle_contribute_help = t("List the people or institutions that participated in creating the resource. Each contributor requires a VCard.<br />The <em>authors</em> are the people or institutions that actually created the resource.<br />Each person or institution that contributed to the resource is designated by the <em>editor</em> role.<br />Each institution that is responsible for the publication of the resource is designated by the <em>publisher</em> role.", array(), array('context' => 'archibald:help'));

  return array(
    // General fields.
    'lomch_general_description' => array(
      'label' => "General description",
      'cardinality' => 1,
      'type' => 'text_long',
      'translatable' => 1,
      'instance_settings' => array(
        'archibald_lom_required' => 1,
      ),
      'widget' => array(
        'type' => 'text_textarea',
      ),
      'api_name' => array(
        'LOM-CHv1.1' => 'general.description',
        // We support only a single description, so we always get key '0' for
        // the value.
        'LOM-CHv1.2' => 'general.description.0',
        'LOM-CHv1.3' => 'general.description.0',
      ),
      'api_help' => array(
        'help' => t("Briefly summarize the content of the resource (about 2 to 6 sentences).", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_general_identifier' => array(
      'label' => "Identifier",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'archibald_lomch_udi_identifier',
      'translatable' => 1,
      'instance_settings' => array(
        'show_title' => 1,
        'archibald_lom_required' => 1,
      ),
      'widget' => array(
        'type' => 'archibald_lomch_udi_identifier',
      ),
      'api_name' => 'general.identifier*',
      'api_help' => array(
        'label' => t("Unique resource identifier(s)", array(), array('context' => 'archibald:help')),
        'help' => t("Identifiers are used to point a user to the described resource itself. This could be a link to a website (URL), a book reference (ISBN), or a digital object identifier (DOI). You may add more than one identifier, and use identifiers of different types.<br />Warning, parts of a resource should not be added as distinct identifiers (e.g., a list of URLs for some videos, which are all part of the described resource itself).", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_resource_language' => array(
      'label' => "Language of the resource",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'taxonomy_term_reference',
      'settings' => array(
        'options_list_callback' => 'archibald_lomch_resource_taxonomy_options_list',
        'allowed_values' => array(
          array(
            'vocabulary' => 'archibald_lomch_description_languages',
            'parent' => 0,
          ),
        ),
      ),
      'instance_settings' => array(
        'archibald_lom_required' => 1,
      ),
      'display' => array(
        'default' => array(
          'type' => 'taxonomy_term_reference_plain',
        ),
      ),
      'widget' => array(
        'type' => 'options_buttons',
      ),
      'api_name' => 'general.language',
      'api_help' => array(
        'help' => t("Select the language(s) in which the resource is available.", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_aggregation_level' => array(
      'label' => "Aggregation level",
      'cardinality' => 1,
      'type' => 'archibald_lomch_ontology_vocabulary',
      'settings' => array(
        'allowed_values_function' => '',
        'allowed_values' => array(),
      ),
      'instance_settings' => array(
        'ontology_vocabulary_id' => 'aggregation_level',
        'archibald_lom_recommended' => 1,
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'widget' => array(
        'type' => 'options_select',
      ),
      'api_name' => 'general.aggregationLevel',
      'api_help' => array(
        'help' => t("Indicate the granularity of the resource. The meaning of each granularity level is as follows:<ol><li>Primary, non-divisible resource. E.g., a picture of the Mona Lisa, or fill-the-blanks text about rodents.</li><li>Teaching sequence. Multiple, non-divisible parts form a sequence. E.g., a video about rodents, with a related fill-the-blanks exercise.</li><li>Collection of teaching sequences. E.g., several related teaching sequences, which, as a whole, can form a course of several lessons.</li><li>Complete course, or several teaching sequences. E.g., a full-scale course, with a certification at the end.</li></ol>", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_keywords' => array(
      'label' => "Keywords",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'taxonomy_term_reference',
      'settings' => array(
        'allowed_values' => array(
          array(
            'vocabulary' => 'archibald_lomch_keywords',
            'parent' => 0,
          ),
        ),
      ),
      'translatable' => 1,
      'instance_settings' => array(
        'archibald_lom_recommended' => 1,
      ),
      'widget' => array(
        'type' => 'taxonomy_autocomplete',
      ),
      'api_name' => 'general.keyword',
      'api_help' => array(
        'help' => t("You can describe the resource using keywords. Please do not use keywords that refer to information that is provided using other fields, like school level, age, author, etc.<br />Suggestion is 3 to 9 keywords, separated by a comma.", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_coverage' => array(
      'label' => "Coverage",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'taxonomy_term_reference',
      'settings' => array(
        'allowed_values' => array(
          array(
            'vocabulary' => 'archibald_lomch_coverage',
            'parent' => 0,
          ),
        ),
      ),
      'translatable' => 1,
      'widget' => array(
        'type' => 'options_select',
        'settings' => array(
          // If the Chosen module exists, activate it for our field.
          'apply_chosen' => 1,
        ),
      ),
      'api_name' => 'general.coverage',
      'api_help' => array(
        'help' => t("The coverage describes the geographic location, time period, and culture to which the resource relates.<br />Some examples:<ul><li>France</li><li>19th century</li><li>Italian Renaissance</li></ul>", array(), array('context' => 'archibald:help')),
      ),
    ),

    // Lifecycle fields.
    'lomch_version' => array(
      'label' => "Version",
      'cardinality' => 1,
      'type' => 'text',
      'translatable' => 1,
      'instance_settings' => array(
        'archibald_lom_recommended' => 1,
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'widget' => array(
        'type' => 'text_textfield',
      ),
      'api_name' => 'lifeCycle.version',
      'api_help' => array(
        'help' => t("The version information.<br />Some examples:<ul><li>Bern, 2015</li><li>Version 2.3, August 2016</li></ul>", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_lifecycle_author' => array(
      'label' => "Authors",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'entityreference',
      'settings' => array(
        'target_type' => 'archibald_vcard',
        'handler_settings' => array(),
      ),
      'instance_settings' => array(
        'archibald_lom_recommended' => 1,
      ),
      'widget' => array(
        'settings' => array(
          'references_dialog_add' => 1,
          'references_dialog_search' => 1,
        ),
        'type' => 'entityreference_autocomplete',
      ),
      'api_name' => 'lifeCycle.contribute*',
      'api_help' => array(
        'label' => $lifecycle_contribute_help_label,
        'help' => $lifecycle_contribute_help,
      ),
    ),
    'lomch_lifecycle_editor' => array(
      'label' => "Editors",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'entityreference',
      'settings' => array(
        'target_type' => 'archibald_vcard',
        'handler_settings' => array(),
      ),
      'instance_settings' => array(
        'archibald_lom_recommended' => 1,
      ),
      'widget' => array(
        'settings' => array(
          'references_dialog_add' => 1,
          'references_dialog_search' => 1,
        ),
        'type' => 'entityreference_autocomplete',
      ),
      'api_name' => 'lifeCycle.contribute*',
      'api_help' => array(
        'label' => $lifecycle_contribute_help_label,
        'help' => $lifecycle_contribute_help,
      ),
    ),
    'lomch_lifecycle_publisher' => array(
      'label' => "Publishers",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'entityreference',
      'settings' => array(
        'target_type' => 'archibald_vcard',
        'handler_settings' => array(),
      ),
      'instance_settings' => array(
        'archibald_lom_recommended' => 1,
      ),
      'widget' => array(
        'settings' => array(
          'references_dialog_add' => 1,
          'references_dialog_search' => 1,
        ),
        'type' => 'entityreference_autocomplete',
      ),
      'api_name' => 'lifeCycle.contribute*',
      'api_help' => array(
        'label' => $lifecycle_contribute_help_label,
        'help' => $lifecycle_contribute_help,
      ),
    ),

    // Technical fields.
    'lomch_technical_format' => array(
      'label' => "Format",
      'cardinality' => 1,
      'type' => 'archibald_lomch_ontology_vocabulary',
      'settings' => array(
        'allowed_values_function' => '',
        'allowed_values' => array(),
      ),
      'instance_settings' => array(
        'ontology_vocabulary_id' => 'technical_format',
        'archibald_lom_recommended' => 1,
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'widget' => array(
        'type' => 'options_select',
      ),
      'api_name' => 'technical.format',
      'api_help' => array(
        'help' => t("If applicable, choose the format in which the resource is available."),
      ),
    ),
    'lomch_size' => array(
      'label' => "Size",
      'cardinality' => 1,
      'type' => 'archibald_lomch_filesize',
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'widget' => array(
        'type' => 'archibald_lomch_filesize',
      ),
      'api_name' => 'technical.size',
      'api_help' => array(
        'help' => t("The file size of the resource. This value can contain decimal values. Try using the most appropriate unit. E.g., prefer using 1GB instead of 1000MB.", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_location' => array(
      'label' => "Location",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'text',
      'widget' => array(
        'type' => 'text_textfield',
      ),
      'instance_settings' => array(
        'archibald_lom_recommended' => 1,
      ),
      'api_name' => 'technical.location',
      'api_help' => array(
        'help' => t("The 'physical' location of the resource. This is usually a URL to the resource. In case of a book, an indication of its location in a library. Multiple locations are possible.", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_platform_requirements' => array(
      'label' => "Platform requirements",
      'cardinality' => 1,
      'type' => 'text_long',
      'translatable' => 1,
      'instance_settings' => array(
        'archibald_lom_recommended' => 1,
      ),
      'widget' => array(
        'type' => 'text_textarea',
      ),
      'api_name' => 'technical.otherPlatformRequirements',
      'api_help' => array(
        'help' => t("Any special technical requirements for using this resource. Common software or equipment does not need to be listed.<br />Some examples:<ul><li>Requires an educanet2 account</li><li>Requires QuickTime version 7.7 or higher</li></ul>", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_duration' => array(
      'label' => "Duration",
      'cardinality' => 1,
      'display' => array(
        'default' => array(
          'label' => 'inline',
          'type' => 'timeperiod_default',
          'settings' => array(
            'granularity' => 3,
          ),
        ),
      ),
      'type' => 'number_integer',
      'widget' => array(
        'type' => 'timeperiod_select',
        'settings' => array(
          // If the Chosen module exists, deactivate it for our field.
          'apply_chosen' => 0,
          'units' => array(
            '3600' => array('max' => 24, 'step size' => 1),
            '60' => array('max' => 59, 'step size' => 1),
            '1' => array('max' => 59, 'step size' => 1),
          )
        ),
      ),
      'api_name' => 'technical.duration',
      'api_help' => array(
        'help' => t("Specify the time required to perform the activity of the resource, like playing a video, or doing an exercise.", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_preview_image' => array(
      'label' => "Preview image",
      'cardinality' => 1,
      'type' => 'image',
      'instance_settings' => array(
        'title_field' => 1,
        'archibald_lom_recommended' => 1,
      ),
      'display' => array(
        'default' => array(
          'settings' => array(
            'image_style' => 'medium',
          ),
        ),
      ),
      'widget' => array(
        'type' => 'image_image',
      ),
      'api_name' => 'technical.previewImage',
      'api_help' => array(
        'help' => t("Provide an image representing the resource. Be careful with work that is protected by copyright.", array(), array('context' => 'archibald:help')),
      ),
    ),

    // Education fields.
    'lomch_learning_resource_type' => array(
      'label' => "Learning resource type",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'archibald_lomch_ontology_vocabulary',
      'settings' => array(
        'allowed_values_function' => '',
        'allowed_values' => array(),
      ),
      'instance_settings' => array(
        'ontology_vocabulary_id' => 'learning_resourcetype',
        'archibald_lom_required' => 1,
      ),
      'widget' => array(
        'type' => 'options_buttons',
      ),
      'api_name' => array(
        'LOM-CHv1.1' => 'education.learningResourceType*',
        // We only support 1 education entry, meaning we will always fetch
        // information for key '0'.
        'LOM-CHv1.2' => 'education.0.learningResourceType*',
        'LOM-CHv1.3' => 'education.0.learningResourceType*',
      ),
      'api_help' => array(
        'help' => t("The <em>documentary</em> type defines the 'physical' format of the resource.<br />The <em>pedagogical</em> type defines the pedagogical characteristics of the resource.", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_intended_enduserrole' => array(
      'label' => "Intended end user role",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'archibald_lomch_ontology_vocabulary',
      'settings' => array(
        'allowed_values_function' => '',
        'allowed_values' => array(),
      ),
      'instance_settings' => array(
        'ontology_vocabulary_id' => 'intended_enduserrole',
        'archibald_lom_recommended' => 1,
      ),
      'widget' => array(
        'type' => 'options_buttons',
      ),
      'api_name' => array(
        'LOM-CHv1.1' => 'education.intendedEndUserRole',
        // We only support 1 education entry, meaning we will always fetch
        // information for key '0'.
        'LOM-CHv1.2' => 'education.0.intendedEndUserRole',
        'LOM-CHv1.3' => 'education.0.intendedEndUserRole',
      ),
      'api_help' => array(
        'help' => t("The intended primary audience(s) for the resource.", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_context' => array(
      'label' => "Context",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'archibald_lomch_ontology_vocabulary',
      'settings' => array(
        'allowed_values_function' => '',
        'allowed_values' => array(),
      ),
      'instance_settings' => array(
        'ontology_vocabulary_id' => 'education_context',
        'archibald_lom_required' => 1,
      ),
      'widget' => array(
        'type' => 'options_buttons',
      ),
      'api_name' => array(
        'LOM-CHv1.1' => 'education.context',
        // We only support 1 education entry, meaning we will always fetch
        // information for key '0'.
        'LOM-CHv1.2' => 'education.0.context',
        'LOM-CHv1.3' => 'education.0.context',
      ),
      'api_help' => array(
        'help' => t("The school level(s) this resource mainly applies to.", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_typical_age_range' => array(
      'label' => "Typical age range",
      'cardinality' => 1,
      'type' => 'range_integer',
      'instance_settings' => array(
        'min' => 1,
        'max' => 99,
        'archibald_lom_recommended' => 1,
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'widget' => array(
        'type' => 'range',
      ),
      'api_name' => array(
        'LOM-CHv1.1' => 'education.typicalAgeRange',
        // We only support 1 education entry, meaning we will always fetch
        // information for key '0'.
        'LOM-CHv1.2' => 'education.0.typicalAgeRange',
        'LOM-CHv1.3' => 'education.0.typicalAgeRange',
      ),
      'api_help' => array(
        'help' => t('Enter the age of the intended end users, if the intended end user role is "learners".'),
      ),
    ),
    'lomch_difficulty_level' => array(
      'label' => "Difficulty level",
      'cardinality' => 1,
      'type' => 'archibald_lomch_ontology_vocabulary',
      'settings' => array(
        'allowed_values_function' => '',
        'allowed_values' => array(),
      ),
      'instance_settings' => array(
        'ontology_vocabulary_id' => 'difficulty',
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'widget' => array(
        'type' => 'options_select',
      ),
      'api_name' => array(
        'LOM-CHv1.1' => 'education.difficultyLevel',
        // We only support 1 education entry, meaning we will always fetch
        // information for key '0'.
        'LOM-CHv1.2' => 'education.0.difficultyLevel',
        'LOM-CHv1.3' => 'education.0.difficultyLevel',
      ),
      'api_help' => array(
        'help' => t("The typical difficulty of this resource, in relationship with the intended end user role.", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_typical_learning_time' => array(
      'label' => "Typical learning time",
      'cardinality' => 1,
      'type' => 'archibald_lomch_ontology_vocabulary',
      'settings' => array(
        'allowed_values_function' => '',
        'allowed_values' => array(),
      ),
      'instance_settings' => array(
        'ontology_vocabulary_id' => 'typical_learning_time',
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'widget' => array(
        'type' => 'options_select',
      ),
      'api_name' => array(
        'LOM-CHv1.1' => 'education.typicalLearningTime',
        // We only support 1 education entry, meaning we will always fetch
        // information for key '0'.
        'LOM-CHv1.2' => 'education.0.typicalLearningTime',
        'LOM-CHv1.3' => 'education.0.typicalLearningTime',
      ),
      'api_help' => array(
        'help' => t("The typical amount of time required when using this resource, in relationship with the intended end user role.", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_educational_description' => array(
      'label' => "Educational description",
      'cardinality' => 1,
      'type' => 'text_long',
      'translatable' => 1,
      'instance_settings' => array(
        'archibald_lom_recommended' => 1,
      ),
      'widget' => array(
        'type' => 'text_textarea',
      ),
      'api_name' => array(
        'LOM-CHv1.1' => 'education.description',
        // We only support 1 education entry, meaning we will always fetch
        // information for key '0'. We also support only a single description,
        // so we always get key '0' for the value as well.
        'LOM-CHv1.2' => 'education.0.description.0',
        'LOM-CHv1.3' => 'education.0.description.0',
      ),
      'api_help' => array(
        'help' => t("Describe:<ul><li>how the resource can be used in a teaching environment</li><li>its content</li><li>what pupils can learn from it</li></ul>If possible, provide additional information for teachers (about 2 to 6 sentences).", array(), array('context' => 'archibald:help')),
      ),
    ),

    // Rights fields.
    'lomch_rights_cost' => array(
      'label' => "Cost",
      'cardinality' => 1,
      'type' => 'archibald_lomch_ontology_vocabulary',
      'settings' => array(
        'allowed_values_function' => '',
        'allowed_values' => array(),
      ),
      'instance_settings' => array(
        'ontology_vocabulary_id' => 'rights_cost',
        'archibald_lom_required' => 1,
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'widget' => array(
        'type' => 'options_select',
      ),
      'api_name' => 'rights.cost',
      'api_help' => array(
        'help' => t("Indicate whether using or possessing this resource implies any costs. If this is the case, the amount can be indicated in the <em>general description</em> field.", array(), array('context' => 'archibald:help')),
      ),
    ),
    'lomch_rights_description' => array(
      'label' => "Terms of use",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'taxonomy_term_reference',
      'settings' => array(
        'options_list_callback' => 'archibald_lomch_resource_taxonomy_options_list',
        'allowed_values' => array(
          array(
            'vocabulary' => 'archibald_lomch_description_licenses',
            'parent' => 0,
          ),
        ),
      ),
      'instance_settings' => array(
        'archibald_lom_required' => 1,
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
          'type' => 'taxonomy_term_reference_plain',
        ),
      ),
      'widget' => array(
        'type' => 'options_select',
        'settings' => array(
          // If the Chosen module exists, activate it for our field.
          'apply_chosen' => 1,
        ),
      ),
      'api_name' => 'rights.copyright*',
      'api_help' => array(
        'help' => t("Indicate under which license(s) the resource is distributed.", array(), array('context' => 'archibald:help')),
      ),
    ),

    // Relations fields.
    'lomch_relations'  => array(
      'label' => "Relations",
      'cardinality' => FIELD_CARDINALITY_UNLIMITED,
      'type' => 'field_collection',
      'translatable' => 1,
      'settings' => array(
        'hide_blank_items' => 1,
        'path' => '',
      ),
      'translatable' => 1,
      'widget' => array(
        'type' => 'field_collection_embed',
      ),
      'display' => array(
        'default' => array(
          'settings' => array(
            'edit' => '',
            'delete' => '',
            'add' => '',
          ),
        ),
      ),
      'api_name' => 'relation*',
      'api_help' => array(
        'label' => t("Related resources", array(), array('context' => 'archibald:help')),
        'help' => t("Related resources can be associated teaching materials, distinct parts, etc. These related resources could be links to a website (URL), book references (ISBN), or digital object identifiers (DOI).<br />Provide a short description (2 to 4 sentences) about the related resource, and how it can be used when teaching.", array(), array('context' => 'archibald:help')),
      ),
    ),

    // Misc fields.
    'archibald_publication_catalogs'  => array(
      'label' => "Catalog(s)",
      'cardinality' => 1,
      'type' => 'list_text',
      'settings' => array(
        'allowed_values_function' => '',
        'allowed_values' => array(
          ARCHIBALD_CATALOG_PARTNER => "Partner",
          ARCHIBALD_CATALOG_NATIONAL => "National",
          ARCHIBALD_CATALOG_BOTH => "Partner + national",
        ),
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'required' => 1,
      'default_value' => array(array('value' => ARCHIBALD_CATALOG_BOTH)),
      'widget' => array(
        'type' => 'options_select',
      ),
    ),
    'archibald_publication_partner' => array(
      'label' => "Partner",
      'cardinality' => 1,
      'type' => 'entityreference',
      'settings' => array(
        'target_type' => 'archibald_partner',
        'handler_settings' => array(),
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'widget' => array(
        'type' => 'options_select',
      ),
    ),
  );
}

/**
 * Define our custom archibald_lomch_description content type.
 *
 * Add fields to our custom content type, and group them together using
 * Field Groups, if it is available.
 *
 * @see archibald_archibald_lomch_description_fields()
 * @see archibald_add_field_to_entity()
 */
function archibald_lomch_description_node_type_insert() {
  $entity = 'node';
  $bundle = 'archibald_lomch_description';

  // Don't promote to front page, don't publish by default and create revisions
  // by default. New revisions are not to be published automatically. Don't
  // show the Preview button. Hide submitted information, as it confuses users
  // more than it helps.
  variable_set('node_options_' . $bundle, array('revision_moderation', 'revision'));
  variable_set('node_preview_' . $bundle, DRUPAL_DISABLED);
  variable_set('node_submitted_' . $bundle, DRUPAL_DISABLED);

  // Enable entity translation for our content type. This will allow us to flag
  // certain fields below as translatable, while keeping others in sync between
  // translations.
  module_load_include('inc', 'entity_translation', 'entity_translation.node');
  variable_set('language_content_type_' . $bundle, ENTITY_TRANSLATION_ENABLED);
  variable_set('entity_translation_settings_node__' . $bundle, array(
    'default_language' => ENTITY_TRANSLATION_LANGUAGE_CURRENT,
    'hide_language_selector' => 0,
    'exclude_language_none' => 1,
    'lock_language' => 0,
    'shared_fields_original_only' => 1,
  ));

  // This also means we want to replace the title field (see
  // https://www.drupal.org/project/title for more information). We want the
  // title to be multilingual.
  title_field_replacement_toggle($entity, $bundle, 'title');

  // Define fields.
  foreach (archibald_archibald_lomch_description_fields() as $field_name => $field_options) {
    archibald_add_field_to_entity($entity, $bundle, $field_name, $field_options);
  }

  // Group fields together in vertical tabs.
  $weight = 0;
  foreach (array(
    'group_general' => array(
      'label' => "General",
      'children' => array(
        'lomch_general_description',
        'lomch_general_identifier',
        'lomch_resource_language',
        'lomch_aggregation_level',
        'lomch_keywords',
        'lomch_coverage',
        'title_field',
      ),
      'open' => 1,
      'required_fields' => 1,
    ),
    'group_lifecycle' => array(
      'label' => "Lifecycle",
      'children' => array(
        'lomch_version',
        'lomch_lifecycle_author',
        'lomch_lifecycle_editor',
        'lomch_lifecycle_publisher',
      ),
    ),
    'group_technical' => array(
      'label' => "Technical",
      'children' => array(
        'lomch_technical_format',
        'lomch_size',
        'lomch_location',
        'lomch_platform_requirements',
        'lomch_duration',
        'lomch_preview_image',
      ),
    ),
    'group_education' => array(
      'label' => "Education",
      'children' => array(
        'lomch_learning_resource_type',
        'lomch_intended_enduserrole',
        'lomch_context',
        'lomch_typical_age_range',
        'lomch_difficulty_level',
        'lomch_typical_learning_time',
        'lomch_educational_description',
      ),
    ),
    'group_rights' => array(
      'label' => "Rights",
      'children' => array(
        'lomch_rights_cost',
        'lomch_rights_description',
      ),
    ),
    'group_relation' => array(
      'label' => "Relation",
      'children' => array(
        'lomch_relations',
      ),
    ),
    'group_classification' => array(
      'label' => "Classification",
      'children' => array(),
    ),
    'group_curricula' => array(
      'label' => "Curricula",
      'children' => array(),
    ),
  ) as $group_name => $group_info) {
    // Create a group for the form, but also for the node default view.
    foreach (array(
      'form' => 'tab',
      'default' => 'fieldset',
    ) as $display => $type) {
      $group = (object) array(
        'identifier' => "{$group_name}|{$entity}|{$bundle}|{$display}",
        'group_name' => $group_name,
        'entity_type' => $entity,
        'bundle' => $bundle,
        'mode' => $display,
        'label' => $group_info['label'],
        'children' => $group_info['children'],
        'weight' => $weight,
        'format_type' => $type,
        'format_settings' => array(
          'formatter' => $type == 'tab' ?
            (empty($group_info['open']) ? 'closed' : 'open') :
            // Groups are always collapsed in view mode.
            'collapsed',
          'instance_settings' => array(
            'description' => !empty($group_info['description']) ? $group_info['description'] : '',
            'classes' => str_replace('_', '-', $group_name) . ' field-group-tab',
            'required_fields' => !empty($group_info['required_fields']),
          ),
        ),
      );

      field_group_group_save($group);

      $weight++;
    }
  }

  // Register the group names as translatable. We cannot use t() directly in the
  // group definition, because it would make translations impossible. But, if we
  // don't do this here, potx will not pick these strings up, which makes our
  // lives harder for distributing translations.
  t("General");
  t("Lifecycle");
  t("Technical");
  t("Education");
  t("Rights");
  t("Relation");
  t("Classification");
  t("Curricula");

  // The relations field is actually a field collection. Add the relevant fields
  // to it as well.
  $entity = 'field_collection_item';
  $bundle = 'lomch_relations';
  foreach (array(
    'lomch_relations_kind' => array(
      'label' => "Kind of relation",
      'cardinality' => 1,
      'type' => 'archibald_lomch_ontology_vocabulary',
      'translatable' => 1,
      'settings' => array(
        'allowed_values_function' => '',
        'allowed_values' => array(),
      ),
      'instance_settings' => array(
        'ontology_vocabulary_id' => 'rel_kind',
        'archibald_lom_required' => 1,
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'widget' => array(
        'type' => 'options_select',
      ),
    ),
    'lomch_relations_identifier' => array(
      'label' => "Identifier",
      'cardinality' => 1,
      'type' => 'archibald_lomch_udi_identifier',
      'translatable' => 1,
      'instance_settings' => array(
        'show_title' => 0,
        'archibald_lom_required' => 1,
      ),
      'display' => array(
        'default' => array(
          'label' => 'inline',
        ),
      ),
      'widget' => array(
        'type' => 'archibald_lomch_udi_identifier',
      ),
    ),
    'lomch_relations_description' => array(
      'label' => "Description",
      'cardinality' => 1,
      'type' => 'text_long',
      'translatable' => 1,
      'instance_settings' => array(
        'archibald_lom_recommended' => 1,
      ),
      'widget' => array(
        'type' => 'text_textarea',
      ),
    ),
  ) as $field_name => $field_options) {
    archibald_add_field_to_entity($entity, $bundle, $field_name, $field_options);
  }
}

/**
 * Add fields to our custom archibald_vcard entity.
 */
function archibald_vcard_entity_insert() {
  $entity = 'archibald_vcard';
  $bundle = 'archibald_vcard';

  // Define fields.
  foreach (array(
    'vcard_first_name' => array(
      'label' => "First name",
      'cardinality' => 1,
      'type' => 'text',
      'settings' => array(
        'max_length' => 50,
      ),
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'vcard_last_name' => array(
      'label' => "Last name",
      'cardinality' => 1,
      'type' => 'text',
      'settings' => array(
        'max_length' => 50,
      ),
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'vcard_organization' => array(
      'label' => "Organization",
      'cardinality' => 1,
      'type' => 'text',
      'settings' => array(
        'max_length' => 100,
      ),
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'vcard_country' => array(
      'label' => "Country",
      'cardinality' => 1,
      'type' => 'text',
      'settings' => array(
        'max_length' => 100,
      ),
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'vcard_zip' => array(
      'label' => "ZIP",
      'cardinality' => 1,
      'type' => 'text',
      'settings' => array(
        'max_length' => 100,
      ),
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'vcard_city' => array(
      'label' => "City",
      'cardinality' => 1,
      'type' => 'text',
      'settings' => array(
        'max_length' => 100,
      ),
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'vcard_address' => array(
      'label' => "Address",
      'cardinality' => 1,
      'type' => 'text',
      'settings' => array(
        'max_length' => 100,
      ),
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'vcard_add_address' => array(
      'label' => "Additional address",
      'cardinality' => 1,
      'type' => 'text',
      'settings' => array(
        'max_length' => 100,
      ),
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'vcard_email' => array(
      'label' => "Email",
      'cardinality' => 1,
      'type' => 'email',
      'settings' => array(
        'max_length' => 100,
      ),
      'widget' => array(
        'type' => 'email_textfield',
      ),
    ),
    'vcard_url' => array(
      'label' => "URL",
      'cardinality' => 1,
      'type' => 'url',
      'settings' => array(
        'max_length' => 255,
      ),
      'widget' => array(
        'type' => 'url_external',
      ),
    ),
    'vcard_phone' => array(
      'label' => "Work phone",
      'cardinality' => 1,
      'type' => 'text',
      'settings' => array(
        'max_length' => 100,
      ),
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'vcard_logo' => array(
      'label' => "Logo",
      'cardinality' => 1,
      'type' => 'image',
      'instance_settings' => array(
        'title_field' => 1,
      ),
      'widget' => array(
        'type' => 'image_image',
      ),
    ),
  ) as $field_name => $field_options) {
    archibald_add_field_to_entity($entity, $bundle, $field_name, $field_options);
  }
}

/**
 * Add fields to our custom archibald_partner entity.
 */
function archibald_partner_entity_insert() {
  $entity = 'archibald_partner';
  $bundle = 'archibald_partner';

  // Define fields.
  foreach (array(
    'partner_display_name' => array(
      'label' => "Display name",
      'cardinality' => 1,
      'type' => 'text',
      'required' => TRUE,
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'partner_username' => array(
      'label' => "Username",
      'cardinality' => 1,
      'type' => 'text',
      'required' => TRUE,
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'partner_privatekey' => array(
      'label' => "Private key",
      'cardinality' => 1,
      'type' => 'file',
      'required' => TRUE,
      'settings' => array(
        'uri_scheme' => 'private',
      ),
      'instance_settings' => array(
        'file_extensions' => 'txt pem key',
      ),
      'widget' => array(
        'type' => 'file_file',
      ),
    ),
    'partner_organization' => array(
      'label' => "Organization",
      'cardinality' => 1,
      'type' => 'text',
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'partner_country' => array(
      'label' => "Country",
      'cardinality' => 1,
      'type' => 'text',
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'partner_zip' => array(
      'label' => "ZIP",
      'cardinality' => 1,
      'type' => 'text',
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'partner_city' => array(
      'label' => "City",
      'cardinality' => 1,
      'type' => 'text',
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'partner_address' => array(
      'label' => "Address",
      'cardinality' => 1,
      'type' => 'text',
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'partner_add_address' => array(
      'label' => "Additional address",
      'cardinality' => 1,
      'type' => 'text',
      'widget' => array(
        'type' => 'text_textfield',
      ),
    ),
    'partner_email' => array(
      'label' => "Email",
      'cardinality' => 1,
      'type' => 'email',
      'widget' => array(
        'type' => 'email_textfield',
      ),
    ),
    'partner_url' => array(
      'label' => "URL",
      'cardinality' => 1,
      'type' => 'url',
      'widget' => array(
        'type' => 'url_external',
      ),
    ),
    'partner_logo' => array(
      'label' => "Logo",
      'cardinality' => 1,
      'type' => 'image',
      'instance_settings' => array(
        'title_field' => 0,
      ),
      'widget' => array(
        'type' => 'image_image',
      ),
    ),
  ) as $field_name => $field_options) {
    archibald_add_field_to_entity($entity, $bundle, $field_name, $field_options);
  }
}

/**
 * Setup the field translation for the taxonomy entities.
 */
function archibald_taxonomy_entity_insert() {
  module_load_include('inc', 'entity_translation', 'entity_translation.node');
  $vocabularies = array(
    'archibald_lomch_coverage' => TRUE,
    'archibald_lomch_keywords' => TRUE,
    'archibald_lomch_description_licenses' => TRUE,
  );

  // Enable the taxonomy translations.
  variable_set('entity_translation_taxonomy', $vocabularies);

  foreach (array_keys($vocabularies) as $bundle) {
    // Enable entity translation for our taxonomies. This will allow us to flag
    // certain fields below as translatable, while keeping others in sync between
    // translations.
    variable_set('entity_translation_settings_taxonomy_term__' . $bundle, array(
      'default_language' => ENTITY_TRANSLATION_LANGUAGE_CURRENT,
      'hide_language_selector' => 0,
      'exclude_language_none' => 0,
      'lock_language' => 0,
      'shared_fields_original_only' => 1,
    ));

    // This also means we want to replace the name field (see
    // https://www.drupal.org/project/title for more information). We want the
    // name to be multilingual.
    $instance = field_info_instance('taxonomy_term', 'name_field', $bundle);
    if (empty($instance)) {
      title_field_replacement_toggle('taxonomy_term', $bundle, 'name');
    }
  }
}

/**
 * Add fields to the core user entity.
 */
function archibald_user_entity_insert() {
  $entity = 'user';
  $bundle = 'user';

  // Define fields.
  foreach (array(
    'user_vcard' => array(
      'label' => "User VCard",
      'cardinality' => 1,
      'type' => 'entityreference',
      'settings' => array(
        'target_type' => 'archibald_vcard',
        'handler_settings' => array(),
      ),
      'widget' => array(
        'settings' => array(
          'references_dialog_add' => 1,
          'references_dialog_search' => 1,
        ),
        'type' => 'entityreference_autocomplete',
      ),
    ),
  ) as $field_name => $field_options) {
    archibald_add_field_to_entity($entity, $bundle, $field_name, $field_options);
  }
}

/**
 * Helper function to add a field to a content type.
 *
 * @param string $entity
 *    The entity type, like 'node'.
 * @param string $bundle
 *    The entity bundle, like 'archibald_vcard'.
 * @param string $field_name
 *    The field machine name.
 * @param array $field_options
 *    The options for the field and field instance.
 */
function archibald_add_field_to_entity($entity, $bundle, $field_name, $field_options) {
  $field = field_info_field($field_name);
  if (empty($field)) {
    field_create_field(array(
      'field_name' => $field_name,
      'cardinality' => $field_options['cardinality'],
      'type' => $field_options['type'],
      'translatable' => !empty($field_options['translatable']) ? $field_options['translatable'] : 0,
      'settings' => !empty($field_options['settings']) ? $field_options['settings'] : array(),
    ));
  }

  $instance = field_info_instance($entity, $field_name, $bundle);
  if (empty($instance)) {
    field_create_instance(array(
      'entity_type' => $entity,
      'bundle' => $bundle,
      'field_name' => $field_name,
      'label' => $field_options['label'],
      'description' => !empty($field_options['description']) ? $field_options['description'] : '',
      'settings' => !empty($field_options['instance_settings']) ? $field_options['instance_settings'] : array(),
      'default_value' => !empty($field_options['default_value']) ? $field_options['default_value'] : NULL,
      'required' => !empty($field_options['required']),
      'display' => !empty($field_options['display']) ? $field_options['display'] : array(),
      'widget' => $field_options['widget'],
    ));
  }
}
