<?php

/**
 * @file
 * Contains archibald_stats_views_query.
 */

class archibald_stats_views_query extends views_plugin_query {

  protected $params = array();
  protected $filters = array();

  /**
   * @{inheritdoc}
   */
  function option_definition() {
    $options = parent::option_definition();
    $options['aggregation_method'] = array(
      'default' => 'day',
    );
    return $options;
  }

  /**
   * @{inheritdoc}
   */
  function options_form(&$form, &$form_state) {
    parent::options_form($form, $form_state);

    $form['aggregation_method'] = array(
      '#type' => 'radios',
      '#title' => t("Statistics aggregation method"),
      '#description' => t("How should the statistics be aggregated by the REST API."),
      '#options' => array(
        'day' => t("By day", array(), array('context' => 'archibald_stats:views')),
        'month' => t("By month", array(), array('context' => 'archibald_stats:views')),
        'year' => t("By year", array(), array('context' => 'archibald_stats:views')),
      ),
      '#default_value' => $this->options['aggregation_method'],
    );
  }

  /**
   * @{inheritdoc}
   */
  function build(&$view) {
    // Store the view in the object to be able to use it later.
    $this->view = $view;

    $view->init_pager();

    // Let the pager modify the query to add limits.
    $this->pager->query();
  }

  /**
   * @{inheritdoc}
   */
  public function alter(&$view) {
    foreach (module_implements('views_query_alter') as $module) {
      $function = $module . '_views_query_alter';
      $function($view, $this);
    }
  }

  /**
   * @{inheritdoc}
   */
  public function execute(&$view) {
    try {
      $start = microtime(TRUE);

      // By default, we fetch data for all partners we know of.
      if (empty($this->params['partner_id'])) {
        $this->params['partner_id'] = array_keys(archibald_partner_load_all());
      }

      // If no dates are provided, fetch from today until one year ago.
      if (!isset($this->params['to'])) {
        $this->params['to'] = time();
      }
      if (!isset($this->params['from'])) {
        $this->params['from'] = strtotime('-1 year', strtotime($this->params['to']));
      }

      $result = array();
      $fields = $this->fields;
      foreach ($this->params['partner_id'] as $partner_id) {
        // Add the results to the array. The results don't contain the partner
        // ID, but if it's part of the fields, we add it now.
        $result += array_map(function($item) use($fields, $partner_id) {
          if (in_array('partner', $fields)) {
            $item['partner'] = $partner_id;
          }
          return $item;
        }, archibald_stats_fetch($partner_id, $this->params['from'], $this->params['to'], $this->options['aggregation_method']));
      }

      if (!empty($result)) {
        // Store raw response.
        $this->api_raw_results = $result;

        // Store results, casting them to objects.
        $view->result = array_map(function($item) {
          return (object) $item;
        }, $result);

        // Do we have to filter?
        if (!empty($this->filters)) {
          foreach ($this->filters as $filter_name => $callback) {
            $view->result = array_filter($view->result, $callback);
          }
        }

        // Fake the pagination. The API doesn't support it for the statistics.
        if (isset($this->offset) && isset($this->limit)) {
          $view->result = array_slice($view->result, $this->offset, $this->limit);
        }

        // Update pager.
        $this->pager->total_items = $view->total_rows = count($view->result);
        $this->pager->update_page_info();
      }
    }
    catch (Exception $e) {
      $view->result = array();
      $view->total_rows = 0;
      if (!empty($view->live_preview)) {
        drupal_set_message($e->getMessage(), 'error');
      }
      else {
        vpr('Exception in @human_name[@view_name]: @message', array('@human_name' => $view->human_name, '@view_name' => $view->name, '@message' => $e->getMessage()));
      }
    }

    $view->execute_time = microtime(TRUE) - $start;
  }

  /**
   * @{inheritdoc}
   */
  function add_relationship($left_table, $left_field, $right_table, $right_field) {
    $this->ensure_table($right_table);
    $this->add_field($left_table, $left_field);
    $this->joins[$left_table][$left_field] = array(
      'table' => $right_table,
      'field' => $right_field,
    );
    return FALSE;
  }

  /**
   * @{inheritdoc}
   */
  public function add_field($table_alias, $field, $alias = '', $params = array()) {
    // Make sure an alias is assigned.
    $alias = $alias ? $alias : $field;
    $this->fields[$alias] = $field;
    return $alias;
  }

  /**
   * @{inheritdoc}
   * @noop
   */
  function add_where($group, $field, $value = NULL, $operator = NULL) { }

  /**
   * @{inheritdoc}
   * @noop
   */
  function set_where_group($type = 'AND', $group = NULL, $where = 'where') {
    return NULL;
  }

  /**
   * @{inheritdoc}
   * @noop
   */
  function ensure_table($table, $relationship = NULL, $join = NULL) { }

  /**
   * Add a parameter for the request.
   *
   * @param string $name
   *    The name of the parameter to add, like "partner_id" or "from".
   * @param * $value
   *    The value of the parameter.
   */
  public function add_parameter($name, $value) {
    $this->params[$name] = $value;
  }

  /**
   * Add a filter for the request.
   *
   * @param string $name
   *    The name of the filter to add.
   * @param callable $callback
   *    The callback that must be applied to the result set in order to filter
   *    it. Check array_filter() for more information.
   */
  public function add_filter($name, $callback) {
    $this->filters[$name] = $callback;
  }

}
