
(function($, Archibald) {

// Prepare a global singleton, which will be an instance of
// ArchibaldCurrMapping.
var archibaldCurrMapping;

var ArchibaldCurrMapping = function(settings) {
  // Store a local reference to the Drupal settings, for easier access.
  this.settings = settings;
};

ArchibaldCurrMapping.prototype = {
  /**
   * All apps that can be the target of suggestions are registered here.
   */
  apps: {},

  /**
   * Register an app as a target for suggestions.
   *
   * @param {ArchibaldCurriculum.Core} app
   */
  registerApp: function(app) {
    if (typeof app.curriculum !== 'undefined') {
      if (typeof this.apps[app.curriculum] === 'undefined') {
        this.apps[app.curriculum] = [];
      }
      this.apps[app.curriculum].push(app);
    }
  },

  /**
   * Can this curriculum receive suggestions, and if so, from whom?
   *
   * @param {String} curriculum
   *    The curriculum to check.
   *
   * @returns {Array}
   *    The list of curricula this curriculum can receive suggestions from.
   */
  canReceiveSuggestionsFrom: function(curriculum) {
    var from = [];
    if (
      typeof this.settings.activateSuggestions[curriculum] !== 'undefined' &&
      typeof this.settings.activateSuggestions[curriculum].providesFor !== 'undefined'
    ) {
      for (var source in this.settings.activateSuggestions[curriculum].providesFor) {
        from.push({
          name: source,
          label: this.settings.activateSuggestions[curriculum].providesFor[source]
        });
      }
    }
    return from.length ? from : false;
  },

  /**
   * Can this curriculum provide suggestions, and if so, for whom?
   *
   * @param {String} curriculum
   *    The curriculum to check.
   *
   * @returns {Array}
   *    The list of curricula this curriculum can provide suggestions for.
   */
  canProvideSuggestionsFor: function(curriculum) {
    var providesFor = [];
    for (var receiver in this.settings.activateSuggestions) {
      if (
        typeof this.settings.activateSuggestions[receiver].providesFor[curriculum] !== 'undefined'
      ) {
        providesFor.push({
          name: receiver,
          label: this.settings.activateSuggestions[receiver].label
        });
      }
    }
    return providesFor.length ? providesFor : false;
  },

  /**
   * Helper method to build the suggestion URL.
   *
   * In order to fetch suggestions, we need to craft a specific URL. This method
   * helps with this.
   *
   * @param {ArchibaldCurriculum.Core} app
   *    The app that will provide the suggestions.
   * @param {ArchibaldCurriculum.ItemModel} model
   *    The item that was just activated. Suggestions will be based on it.
   *
   * @returns {String}
   *    An URL for an ajax request to fetch suggestions.
   */
  getSuggestionURL: function(app, model) {
    return this.settings.suggestionURL + '/' +
      app.curriculum + '/' +
      _.map(app.providesFor, function(item) {
        return item.name;
      }).join(',') + '/' +
      model.get('id');
  },

  /**
   * Add the suggestion to the concerned apps.
   *
   * @param {String} curriculum
   *    The curriculum that the suggestion was for. Target all apps that are for
   *    this curriculum.
   * @param {String} itemId
   *    The item to activate in all concerned apps.
   */
  addSuggestion: function(curriculum, itemId) {
    var app, model;
    if (typeof this.apps[curriculum] !== 'undefined') {
      for (var i = 0, len = this.apps[curriculum].length; i < len; i++) {
        app = this.apps[curriculum][i];
        model = app.getItemDatabase().findWhere({
          id: itemId
        });

        if (model && !model.get('active')) {
          model.set('active', true);

          // Because this was not done through the UI, the app won't trigger
          // the recursive checking logic. Trigger it here.
          app.recursiveCheck(model);
        }
      }
    }
  }
};

Drupal.behaviors.archibaldCurricula.hooks.appRegistered.push(function(app) {
  // Use this opportunity to make sure we have a global ArchibaldCurrMapping
  // object.
  if (typeof archibaldCurrMapping === 'undefined') {
    archibaldCurrMapping = new ArchibaldCurrMapping(
      Drupal.settings.archibaldCurrMapping
    );
  }

  var from, providesFor;
  if (typeof app.curriculum !== 'undefined') {
    from = archibaldCurrMapping.canReceiveSuggestionsFrom(app.curriculum);
    providesFor = archibaldCurrMapping.canProvideSuggestionsFor(app.curriculum);
  }

  // Is this an app that can "receive" suggestions?
  if (from) {
    archibaldCurrMapping.registerApp(app);
  }

  // Is this an app with which we can provide suggestions?
  if (providesFor) {
    app.providesFor = providesFor;
    app.optOutOfSuggestions = {};

    // Activate the suggestions for this editor.
    app.on('app:render', function() {
      // Add an option to opt out of the suggestions.
      app.getWrapper().find('.archibald-curricula-field__editor__opts').append(
        _.template(Archibald.templates.currMappingSuggestionsOpt)({
          items: providesFor
        })
      );

      // Opt out of suggestions dialog logic.
      app.getWrapper()
        .find('.archibald-curricula-field__editor__opts__opt--suggestions-opt-out input')
        .change(function() {
          var $this = $(this);
          app.optOutOfSuggestions[$this.attr('data-for')] = !$this.is(':checked');
        })
        .change();

      // Add a wrapper to contain the suggestions.
      app.getWrapper().find('.archibald-curriculum-ui__editor').after(
        _.template(Archibald.templates.currMappingSuggestionsWrapper)()
      );

      // Store the suggestion wrapper "globally" by attaching it to the app
      // object, and provide shortcuts for the child elements as well.
      app.$suggestionWrapper = app.getWrapper().find('.archibald-curricula-field__editor__suggestions');
      app.$suggestionWrapper.$dismiss = app.$suggestionWrapper.find('.archibald-curricula-field__editor__suggestions__close');
      app.$suggestionWrapper.$content = app.$suggestionWrapper.find('.archibald-curricula-field__editor__suggestions__content');

      // Add click events for the "dismiss" link, and trigger it immediately.
      app.$suggestionWrapper.$dismiss.click(function() {
        app.$suggestionWrapper.$content.empty();
        app.$suggestionWrapper.slideUp();
      }).click();
    });
  }
});

Drupal.behaviors.archibaldCurricula.hooks.appReady.push(function(app) {
  // Is this an app for which we can provide suggestions?
  if (
    typeof app.providesFor !== 'undefined' &&
    app.providesFor
  ) {
    // Show suggestions when an item is activated.
    app.on('column:item:change', function(model, collection, column) {
      if (model.get('active')) {
        // Fetch suggestions. Make sure we only fetch one payload at a time.
        // if there was already an ajax request being made, cancel it.
        if (
          typeof app.suggestionRequest !== 'undefined'
        ) {
          app.suggestionRequest.abort();
        }

        app.suggestionRequest = $.ajax({
          url: archibaldCurrMapping.getSuggestionURL(app, model),
          dataType: 'json',
          success: function(json) {
            // We consider this request as completed.
            app.suggestionRequest = undefined;

            // Add the labels to the json data, so we don't have to look them up
            // in the template.
            _.map(app.providesFor, function(item) {
              if (typeof json[app.curriculum][item.name] !== 'undefined') {
                json[app.curriculum][item.name].label = item.label;
              }
            });

            // Render them.
            app.$suggestionWrapper.$content.html(
              _.template(Archibald.templates.currMappingSuggestions)({
                items: json[app.curriculum]
              })
            );

            // Handle the "add" links click events.
            app.$suggestionWrapper.$content
              .find('.archibald-curricula-field__editor__suggestions__content__add')
              .click(function() {
              var $this = $(this);

              // Add the suggestion.
              archibaldCurrMapping.addSuggestion(
                $this.attr('data-for'),
                $this.attr('data-item-id')
              );

              // Perform some UI/UX updates. Change the label to "added".
              $this.html(Drupal.t("added"));

              // Wait for 1000ms, and remove the suggestion. If this was the
              // last suggestion, hide the entire suggestion wrapper.
              setTimeout(function() {
                if (!$this.parent().siblings().length) {
                  // There are no more suggestions. Hide everything.
                  app.$suggestionWrapper.slideUp();
                }
                else {
                  // Only hide this suggestion.
                  $this.parent().fadeOut({ complete: function() {
                    $(this).remove();
                  }});
                }
              }, 1000);
            });

            // Show the suggestions.
            app.$suggestionWrapper.slideDown();
          }
        });
      }
    });
  }
});

})(jQuery, window.ArchibaldCurriculum || (window.ArchibaldCurriculum = new Object()));
