<?php

namespace Drupal\entity_reference_revisions\Normalizer;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\entity_reference_revisions\EntityNeedsSaveInterface;
use Drupal\entity_reference_revisions\Plugin\Field\FieldType\EntityReferenceRevisionsItem;
use Drupal\hal\Normalizer\EntityReferenceItemNormalizer;

/**
 * Defines a class for normalizing EntityReferenceRevisionItems.
 */
class EntityReferenceRevisionItemNormalizer extends EntityReferenceItemNormalizer {

  /**
   * The interface or class that this Normalizer supports.
   *
   * @var string
   */
  protected $supportedInterfaceOrClass = EntityReferenceRevisionsItem::class;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Set the entity type manager.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function setEntityTypeManager(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  protected function constructValue($data, $context) {
    $field_item = $context['target_instance'];
    /** @var \Drupal\entity_reference_revisions\Plugin\Field\FieldType\EntityReferenceRevisionsItem $field_item */
    $field_definition = $field_item->getFieldDefinition();
    $target_type_id = $field_definition->getSetting('target_type');
    $target_type = $this->entityTypeManager->getDefinition($target_type_id);

    // If the target type has a revision parent type field and that field is
    // part of data, this is a composite normalization.
    $parent_type_field = $target_type->get('entity_revision_parent_type_field');
    if ($parent_type_field && isset($data[$parent_type_field])) {

      // Denormalize the embedded entity, if it implements the
      // EntityNeedsSaveInterface interface, ensure that it will be saved.
      if ($target_entity = $this->serializer->denormalize($data, $target_type->getClass(), 'hal_json')) {
        return ['entity' => $target_entity];
      }
    }

    $value = parent::constructValue($data, $context);
    if ($value) {
      $value['target_revision_id'] = $data['target_revision_id'];
    }
    return $value;
  }

  /**
   * {@inheritdoc}
   */
  public function normalize($field_item, $format = NULL, array $context = array()) {
    $data = parent::normalize($field_item, $format, $context);
    /** @var \Drupal\entity_reference_revisions\Plugin\Field\FieldType\EntityReferenceRevisionsItem $field_item */
    $target_entity = $field_item->get('entity')->getValue();

    $entity = $field_item->getEntity();
    $field_name = $field_item->getParent()->getName();
    $field_uri = $this->linkManager->getRelationUri($entity->getEntityTypeId(), $entity->bundle(), $field_name, $context);

    if ($target_entity->getEntityType()->get('entity_revision_parent_id_field')) {
      $embedded_context = $context;
      $embedded_context['included_fields'] = NULL;
      $embedded = $this->serializer->normalize($target_entity, $format, $embedded_context);

      // Restore lang key if it was set by our parent.
      if (isset($data['_links'][$field_uri][0]['lang'])) {
        $embedded['lang'] = $data['_links'][$field_uri][0]['lang'];
      }

      // Remove empty links.
      if (empty($data['_links'][$field_uri][0]['href'])) {
        unset($data['_links'][$field_uri]);
      }
      if (empty($embedded['_links']['self']['href'])) {
        unset($embedded['_links']['self']);
      }

      $data['_embedded'][$field_uri][0] = $embedded;
    }
    else {
      $data['_embedded'][$field_uri][0]['target_revision_id'] = $field_item->target_revision_id;
    }
    return $data;
  }

}
