<?php

namespace Drupal\Tests\entity_reference_revisions\Kernel;

use Drupal\Core\Field\FieldStorageDefinitionInterface;
use Drupal\entity_composite_relationship_test\Entity\EntityTestCompositeRelationship;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\node\Entity\Node;
use Drupal\node\Entity\NodeType;
use Drupal\Tests\node\Traits\ContentTypeCreationTrait;
use Drupal\Tests\node\Traits\NodeCreationTrait;

/**
 * Tests the entity_reference_revisions normalizer.
 *
 * @group entity_reference_revisions
 */
class EntityReferenceRevisionsNormalizerTest extends EntityKernelTestBase {

  use ContentTypeCreationTrait;
  use NodeCreationTrait;

  /**
   * Modules to enable.
   *
   * @var array
   */
  public static $modules = [
    'node',
    'field',
    'entity_reference_revisions',
    'entity_composite_relationship_test',
    'language',
    'hal',
    'serialization',
    'rest',
  ];

  /**
   * SerializerInterface which is used for serialization.
   *
   * @var \Symfony\Component\Serializer\Normalizer\NormalizerInterface
   */
  protected $serializer;

  /**
   * {@inheritdoc}
   */
  protected function setUp() {
    parent::setUp();

    $this->installEntitySchema('entity_test_composite');
    $this->installSchema('node', ['node_access']);

    // Create article content type.
    NodeType::create([
      'type' => 'article',
      'name' => 'Article',
    ])->save();

    // Create the reference to the composite entity test for a nested composite
    // entity test.
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'composite_nested_reference',
      'entity_type' => 'entity_test_composite',
      'type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'entity_test_composite',
      ],
    ]);
    $field_storage->save();
    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'entity_test_composite',
      'translatable' => FALSE,
    ]);
    $field->save();

    // Create the reference to the composite entity test.
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'composite_reference',
      'entity_type' => 'node',
      'type' => 'entity_reference_revisions',
      'settings' => [
        'target_type' => 'entity_test_composite',
      ],
    ]);
    $field_storage->save();
    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'article',
      'translatable' => FALSE,
    ]);
    $field->save();

    // Create a non-composite reference to a node.
    $field_storage = FieldStorageConfig::create([
      'field_name' => 'node_reference',
      'entity_type' => 'node',
      'type' => 'entity_reference_revisions',
      'cardinality' => FieldStorageDefinitionInterface::CARDINALITY_UNLIMITED,
      'settings' => [
        'target_type' => 'node',
      ],
    ]);
    $field_storage->save();
    $field = FieldConfig::create([
      'field_storage' => $field_storage,
      'bundle' => 'article',
      'translatable' => FALSE,
    ]);
    $field->save();

    // Inject serializer.
    $this->serializer = $this->container->get('serializer');
  }

  /**
   * Tests the embedded paragraph when normalizing.
   */
  public function testEmbedEntityReference() {
    // Create the test composite entity.
    $nested_composite = EntityTestCompositeRelationship::create([
      'name' => $nested_composite_name = $this->randomMachineName(),
    ]);
    $nested_composite->save();

    $composite = EntityTestCompositeRelationship::create([
      'name' => $composite_name = $this->randomMachineName(),
      'composite_nested_reference' => $nested_composite,
    ]);
    $composite->save();

    // Add 2 nodes for referencing.
    $node_reference = Node::create([
      'type' => 'article',
      'title' => 'Referenced node',
    ]);
    $node_reference->save();
    $node_reference2 = Node::create([
      'type' => 'article',
      'title' => 'Referenced node #2',
    ]);
    $node_reference2->save();

    // Add content with the nested composite entity test and the 2 referenced
    // nodes.
    $node = Node::create([
      'type' => 'article',
      'title' => 'Normalizing paragraph test',
      'node_reference' => [$node_reference, $node_reference2],
      'composite_reference' => $composite,
    ]);
    $node->save();

    // Check the serialization of embedded composite field.
    $normalized = $this->serializer->normalize($node, 'hal_json');

    // Check the id of the composite reference in the node.
    $revision_id = $node->get('composite_reference')->target_revision_id;
    $this->assertEquals($revision_id,
      $normalized['_embedded']['http://localhost/rest/relation/node/article/composite_reference'][0]['id'][0]['value']);

    // Check the id of the nested composite reference in the entity test
    // composite.
    $nested_revision_id = $composite->get('composite_nested_reference')->target_revision_id;
    $this->assertEquals($nested_revision_id,
      $normalized['_embedded']['http://localhost/rest/relation/node/article/composite_reference'][0]['_embedded']['http://localhost/rest/relation/entity_test_composite/entity_test_composite/composite_nested_reference'][0]['id'][0]['value']);

    // Assert there is no link to the embedded composite as it has no href but
    // the one to the node is still there.
    $this->assertFalse(isset($normalized['_links']['http://localhost/rest/relation/node/article/composite_reference']));
    $this->assertFalse(isset($normalized['_embedded']['http://localhost/rest/relation/node/article/composite_reference'][0]['_links']['self']));
    $this->assertTrue(isset($normalized['_links']['http://localhost/rest/relation/node/article/node_reference'][0]['href']));

    // Check the denormalization of the embedded composite field.
    $nested_composite->delete();
    $composite->delete();
    $node->delete();
    $node = $this->serializer->denormalize($normalized, Node::class, 'hal_json');
    $node->save();

    $node = Node::load($node->id());

    // Check the id of the composite reference in the node.
    $this->assertEquals($revision_id, $node->get('composite_reference')->target_revision_id);
    $this->assertEquals($composite_name, $node->get('composite_reference')->entity->label());

    // Check the id of the nested composite reference in the entity test
    // composite.
    $this->assertEquals($nested_revision_id,
      $node->get('composite_reference')->entity->get('composite_nested_reference')->target_revision_id);
    $this->assertEquals($nested_composite_name,
      $node->get('composite_reference')->entity->get('composite_nested_reference')->entity->label());
  }

}
