<?php
// $Id: archive.inc,v 1.1.2.9 2007/08/30 18:14:22 susurrus Exp $

/**
 * * Fetch nodes for the selected date, or current date if none selected.
 *
 * @param $year
 *   Number of year
 * @param $month
 *   Number of month
 * @param $day
 *   Number of day
 * @return
 *   A string with the themed page
 */
function _archive_page($type = 'all', $year = NULL, $month = NULL, $day = NULL) {
  $output = '';
  $date = _archive_date($type, $year, $month, $day);

  $nodes = variable_get('default_nodes_main', 10);
  $query = _archive_query($type, $date);
  $result = pager_query(db_rewrite_sql(array_shift($query)), $nodes, 0, NULL, $query);
  if (db_num_rows($result)) {
    $output = theme('archive_navigation', $type, $date);
    while ($node = db_fetch_object($result)) {
      $output .= node_view(node_load($node->nid), TRUE);
    }
    $output .= theme('pager', NULL, $nodes);
    return $output;
  }
  else if ($date->days[$date->day]) {
    drupal_goto(_archive_url('all', $date->year, $date->month, $date->day));
  }
  else if ($date->months[$date->month]) {
    drupal_goto(_archive_url('all', $date->year, $date->month));
  }
  else if ($date->years[$date->year]) {
    drupal_goto(_archive_url('all', $date->year));
  }
  else if (arg(0) != 'archive') {
    drupal_goto('archive');
  }
  else {
    $output = '<p>'. t('There\'s currently nothing in the archives to browse.') .'</p>';
    return $output;
  }
}

/**
 * Parses the current URL and populates an archive
 * date object with the selected date information.
 *
 * @param $year
 *   Number of year
 * @param $month
 *   Number of month
 * @param $day
 *   Number of day
 * @return
 *   A date object with GMT date values and a timezone value
 */
function _archive_date($type, $year = NULL, $month = NULL, $day = NULL) {

  $date = (object) array('tz' => _archive_get_timezone());

  $date->year  = 0;
  $date->month = 0;
  $date->day   = 0;
  if (_archive_validate_year($year)) {
    $date->year = $year;
    if (_archive_validate_month($month)) {
      $date->month = $month;
      if (_archive_validate_day($year, $month, $day)) {
         $date->day = $day;
      }
    }
  }
  $post_counts = _archive_post_count($type, $date);
  $date->years  = $post_counts['years'];
  $date->months = $post_counts['months'];
  $date->days   = $post_counts['days'];

  return $date;
}

/**
 * Builds an SQL query array (query and parameters) to use
 * to display an archive page for the specified date.
 *
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object obtained from _archive_date()
 * @return
 *   An array of (query, param_start, param_end)
 */
function _archive_query($type, $date) {
  // Confine the display interval to only one day
  if ($date->day) {
    $start = gmmktime(0, 0, 0, $date->month, $date->day, $date->year);
    $end   = gmmktime(0, 0, 0, $date->month, $date->day + 1, $date->year);
  }
  // Confine the display interval to one month
  else if ($date->month) {
    $start = gmmktime(0, 0, 0, $date->month, 1, $date->year);
    $end   = gmmktime(0, 0, 0, $date->month + 1, 1, $date->year);
  }
  // Confine the display interval to one year
  else if ($date->year) {
    $start = gmmktime(0, 0, 0, 1, 1, $date->year);
    $end   = gmmktime(0, 0, 0, 1, 1, $date->year + 1);
  }
  else {
    $start = 0;
    $end = 0;
  }
  
  // Grab limits on node types if exist
  $final_types = _archive_types_sql_string($type);
  
  // Allow viewing all nodes, not just nodes by year
  if ($start && $end) {
    return array('SELECT n.nid, n.type FROM {node} n WHERE n.status = 1 '. $final_types .'AND n.created >= %d AND n.created < %d ORDER BY n.created DESC', $start - $date->tz, $end - $date->tz);
  }
  else {
    return array('SELECT n.nid, n.type FROM {node} n WHERE n.status = 1 '. $final_types .'ORDER BY n.created DESC');
  }
}

/**
 * Builds a SQL statement to check that the appropriate
 * node types are being returned 
 *
 * @param $type
 *   A string representing the node-type currently being displayed
 * @return
 *    A SQL string
 */
function _archive_types_sql_string($type){
  // Validate type and specify node types to include
  $final_types = '';
  if (_archive_validate_type($type) && $type != 'all') {
    $final_types = $type; 
  }
  else {
    $types = variable_get('archive_type_filters', array());
    // If no checkboxes selected
    if (!array_key_exists('0', $types)) {
      foreach ($types as $key => $value) {
        if (!$value) {
          unset($types[$key]);
        }
      }
      $final_types = join(array_keys($types), '", "');
    }
  }
  if (strlen($final_types) > 0) {
    $final_types = 'AND n.type IN ("' . $final_types . '") ';
  }
  return $final_types;
}


/**
 * Returns the count of nodes per day/month/year
 *
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object obtained from _archive_date()
 * @return
 *   An array, indexed by date and with post counts as values.
 */
function _archive_post_count($type, $date) {
  $final_types = _archive_types_sql_string($type);
  $result = db_fetch_object(db_query(db_rewrite_sql('SELECT MIN(n.created) AS min_date, MAX(n.created) AS max_date FROM {node} n WHERE n.status = 1 '. $final_types)));
  $mod = '';
  $with_posts = array('years' => array(), 'months' => array(), 'days' => array());
  $node_query = db_query(db_rewrite_sql('SELECT n.uid, n.created FROM {node} n WHERE n.status = 1 '. $final_types));
  while ($o = db_fetch_object($node_query)) {
    $node_date = explode(' ', format_date($o->created, 'custom', 'Y n j'));
    $with_posts['years'][$node_date[0]]++;
    if ($node_date[0] == $date->year) {
      $with_posts['months'][$node_date[1]]++;
      if ($node_date[1] == $date->month) {
        $with_posts['days'][$node_date[2]]++;
      }
    }
  }
  return $with_posts;
}

/**
 * Returns the different node types that have nodes
 *
 * @param $date
 *    A date object obtained from _archive_date()
 * @return
 *    An array of node-types to number of posts of that type
 */
function _archive_node_types($date) {
  
  $types = variable_get('archive_type_filters', array());
  
  // Confine the display interval to only one day
  if ($date->day) {
    $start = gmmktime(0, 0, 0, $date->month, $date->day, $date->year);
    $end   = gmmktime(0, 0, 0, $date->month, $date->day + 1, $date->year);
  }
  // Confine the display interval to one month
  elseif ($date->month) {
    $start = gmmktime(0, 0, 0, $date->month, 1, $date->year);
    $end   = gmmktime(0, 0, 0, $date->month + 1, 1, $date->year);
  }
  // Confine the display interval to one year
  elseif ($date->year) {
    $start = gmmktime(0, 0, 0, 1, 1, $date->year);
    $end   = gmmktime(0, 0, 0, 1, 1, $date->year + 1);
  }
  else {
    $start = 0;
    $end = 0;
  }
  
  $result = '';
  if ($start && $end) {
    $result = db_query(db_rewrite_sql('SELECT t.type, t.name, COUNT(n.nid) AS node_count FROM {node} n INNER JOIN {node_type} t ON t.type = n.type WHERE n.status = 1 AND t.type IN ("'. join($types, '", "') .'") AND n.created BETWEEN %d AND %d GROUP BY n.type ORDER BY n.created'), $start - $date->tz, $end - $date->tz);
  }
  else {
    $result = db_query(db_rewrite_sql('SELECT t.type, t.name, COUNT(n.nid) AS node_count FROM {node} n INNER JOIN {node_type} t ON t.type = n.type WHERE n.status = 1 AND t.type IN ("'. join($types, '", "') .'") GROUP BY n.type ORDER BY n.created'));
  }
  
  $n_types = array();
  while ($row = db_fetch_array($result)) {
    $n_types[$row['type']] = array('count' => $row['node_count'],
                                   'name'  => $row['name']);
  }
  
  ksort($n_types);
  return $n_types;
}

/**
 * Check if given year is valid for a Drupal archive
 *
 * @param $year
 *    The year to check
 * @return
 *    TRUE or FALSE
 */
function _archive_validate_year($year) {
  return (1990 < $year && $year < 2040);
}

/**
 * Check if given month is valid.
 *
 * @param $month
 *    The month to check
 * @return
 *    TRUE or FALSE
 */
function _archive_validate_month($month) {
  return (1 <= $month && $month <= 12);
}

/**
 * Check if given year, month and date combination
 * is valid for a Drupal archive.
 *
 * @param $year
 *    The year to check
 * @param $month
 *    The month to check
 * @param $day
 *    The day to check
 * @return
 *    TRUE or FALSE
 */
function _archive_validate_day($year, $month, $day) {
  if (_archive_validate_month($month) && _archive_validate_year($year)) {
    // Number of days for that month
    $last = gmdate('t', gmmktime(0, 0, 0, $month, 1, $year));
    return (1 <= $day && $day <= $last);
  }
  else {
    return FALSE;
  }
}

/**
 * Check if given node type is valid for a Drupal archive
 *
 * @param $type
 *   A string representing the node-type currently being displayed
 * @return
 *    TRUE or FALSE
 */
function _archive_validate_type($type) {
  $types = variable_get('archive_type_filters', array());
  foreach ($types as $key => $value) {
    if (!$value) {
      unset($types[$key]);
    }
    else {
      $types[$key] = 0;
    }
  }
  return in_array($type, array_keys($types));
}

/**
 * Generate an archive URL based on the $y, $m and $d
 * provided, falling back on the $date properties if
 * an invalid date is specified.
 *
 * Validation checking included for $date properties,
 * because those could be zero, if not defined from the
 * URL.
 *
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $y
 *    The year to use if valid
 * @param $m
 *    The month to use if valid
 * @param $d
 *    The day to use if valid
 * @return
 *    A string with the generated archive URL
 */
function _archive_url($type, $date, $y = 0, $m = 0, $d = 0) {
  $url = 'archive';

  if (_archive_validate_type($type)) {
    $url .= '/' . $type;
  }
  else {
    $url .= '/all';
  }

  if (_archive_validate_year($y)) {
    $url .= '/'. $y;
    if (_archive_validate_month($m)) {
      $url .= '/'. $m;
      if (_archive_validate_day($y, $m, $d)) {
        $url .= '/'. $d;
      }
    }
  }
  else if (_archive_validate_year($date->year) && _archive_validate_month($m)) {
    $url .= '/'. $date->year .'/'. $m;
    if (_archive_validate_day($date->year, $m, $d)) {
      $url .= '/'. $d;
    }
  }
  else if (_archive_validate_year($date->year) && _archive_validate_month($date->month)) {
    $url .= '/'. $date->year .'/'. $date->month;
    if (_archive_validate_day($date->year, $date->month, $d)) {
      $url .= '/'. $d;
    }
  }
  return $url;
}

/**
 * Determine timezone to use for the dates (from format_date)
 *
 * @return
 *   Timezone offset to use in time operations
 */
function _archive_get_timezone() {
  global $user;
  if (variable_get('configurable_timezones', 1) && $user->uid && strlen($user->timezone)) {
    return $user->timezone;
  }
  else {
    return variable_get('date_default_timezone', 0);
  }
}

/**
 * Theme the archive navigation with years, months and dates by default.
 * 
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object returned from _archive_date()
 */
function theme_archive_navigation($type, $date) {
  $output  = "<div id=\"archive-container\"><dl><dt>". t('Date') ."</dt><dd>\n";
  $output .= theme('archive_navigation_years', $type, $date);
  if (_archive_validate_year($date->year)) {
    $output .= theme('archive_navigation_months', $type, $date);
  }
  if (_archive_validate_month($date->month)) {
    $output .= theme('archive_navigation_days', $type, $date);
  }
  $output .= "</dd>";
  // Only display node type filter if more than one node type represented
  if (count(_archive_node_types($date)) > 1) {
    $output .= "<dt>". t('Type') ."</dt><dd>\n";
    $output .= theme('archive_navigation_node_types', $type, $date);
    $output .= "</dd>";
  }
  $output .= "</dl></div>\n";
  return $output;
}

/**
 * Theme the list of years for the archive navigation.
 * 
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object returned from _archive_date()
 */
function theme_archive_navigation_years($type, $date) {
  $all_years = array_keys($date->years);
  $output = "<ul id=\"archive-years\">\n";
  $all_count = 0;
  foreach ($date->years as $year) {
    $all_count += $year;
  }
  $output .= '<li'. ($date->year?'':' class="selected"') .'>'. l('All', 'archive', array('title' => format_plural($all_count, '1 post', '@count posts'))) ."</li>\n";
  for ($year = min($all_years); $year <= max($all_years); $year++) {
    $class = '';
    if ($year == $date->year) {
      $class = ' class="selected"';
    }
    $output .= '<li'. $class .'>'. l($year, _archive_url($type, $date, $year), array('title' => format_plural($date->years[$year], '1 post', '@count posts'))) ."</li>\n";
  }
  $output .= "</ul>\n";

  return $output;
}

/**
 * Theme the list of months for the archive navigation.
 *
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object returned from _archive_date()
 */
function theme_archive_navigation_months($type, $date) {
  $output = "<ul id=\"archive-months\">\n";
  
  $all_count = 0;
  foreach ($date->months as $month) {
    $all_count += $month;
  }
  
  $output .= '<li'. ($date->month?'':' class="selected"') .'>'. l('All', _archive_url($type, $date, $date->year), array('title' => format_plural($all_count, '1 post', '@count posts'))) ."</li>\n";
  $curr_month = date('n', time());
  $curr_year = date('Y', time());
  foreach (range(1, 12) as $month) {
    $posts = !empty($date->months[$month]) ? $date->months[$month] : 0;
    $class = '';
    if ($month == $date->month) {
      $class = ' class="selected"';
    }
    elseif ($curr_year == $date->year && $month > $curr_month) {
      $class = ' class="future"';
    }
    $month_names = array('', 'Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec');
    $output .= "<li$class>". ($posts > 0 ? l($month_names[$month], _archive_url($type, $date, 0, $month), array('title' => format_plural($posts, "1 post", "@count posts"))) : $month_names[$month]) ."</li>\n";
  }
  $output .= "</ul>\n";
  return $output;
}

/**
 * Theme the list of days for the archive navigation.
 * 
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object returned from _archive_date()
 */
function theme_archive_navigation_days($type, $date) {
  $output = "";
  $day_stop = gmdate('t', gmmktime(0, 0, 0, $date->month, 1, $date->year));
  $output = "<ul id=\"archive-days\">\n";
  
  $all_count = 0;
  foreach ($date->days as $day) {
    $all_count += $day;
  }
  
  $output .= '<li'. ($date->day?'':' class="selected"') .'>'. l('All', _archive_url($type, $date, $date->year, $date->month), array('title' => format_plural($all_count, '1 post', '@count posts'))) ."</li>\n";
  $curr_month = date('n', time());
  $curr_year = date('Y', time());
  $curr_day = date('j', time());
  for ($day = 1; $day <= $day_stop; $day++) {
    $posts = array_key_exists($day, $date->days) ? $date->days[$day] : 0;
    $class = '';
    if ($day == $date->day) {
      $class = ' class="selected"';
    }
    elseif ($curr_year == $date->year && $curr_month == $date->month && $day > $curr_day) {
      $class = ' class="future"';
    }
    $output .= "<li$class>". ($posts ? l($day, _archive_url($type, $date, 0, 0, $day), array("title" => format_plural($posts, "1 post", "@count posts"))) : $day) ."</li>\n";
  }
  $output .= "</ul>\n";
  return $output;
}

/**
 * Theme the list of node types for the archives
 * 
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object returned from _archive_date()
 */
function theme_archive_navigation_node_types($type, $date) {
  $output = "<ul id=\"archive-node_types\">\n";
  $types_count = _archive_node_types($date);
  
  $all_count = 0;
  foreach ($types_count as $t) {
    $all_count += $t['count'];
  }
  
  $output .= '<li'. ($type != 'all'?'':' class="selected"') .'>'. l('All', _archive_url('all', $date), array('title' => format_plural($all_count, '1 post', '@count posts'))) ."</li>\n";
  foreach ($types_count as $ft_key => $ft_value) {
    if (!$ft_value['count']) {
      continue;
    }
    $class = ($ft_key == $type ? ' class="selected"' : '');
    $name = $ft_value['name'];
    if ($types_count[$ft_key]['count'] > 0) {
      $output .= "<li$class>". l($name, _archive_url($ft_key, $date, $date->year, $date->month, $date->day), array("title" => format_plural($types_count[$ft_key]['count'], "1 post", "@count posts"))) ."</li>\n";
    }
    else {
      $output .= "<li$class>$name</li>\n";
    }
  }
  $output .= "</ul>\n";
  return $output;
}

?>