<?php

/**
 * * Fetch nodes for the selected date, or current date if none selected.
 *
 * @param $year
 *   Number of year
 * @param $month
 *   Number of month
 * @param $day
 *   Number of day
 * @return
 *   A string with the themed page
 */
function _archive_page($type = 'all', $year = NULL, $month = NULL, $day = NULL) {
  $output = '';
  $date = _archive_date($type, $year, $month, $day);

  $nodes = variable_get('default_nodes_main', 10);
  $query = _archive_query($type, $date);
  $result = pager_query(db_rewrite_sql(array_shift($query)), $nodes, 0, NULL, $query);
  if (db_num_rows($result)) {
    
    // Set the page title for a proper archive page
    drupal_set_title(theme('archive_page_title', $type, $year, $month, $day));
    
    $output = theme('archive_navigation', $type, $date);
    while ($node = db_fetch_object($result)) {
      $output .= node_view(node_load($node->nid), TRUE);
    }
    $output .= theme('pager', NULL, $nodes);
    return $output;
  }
  else if ($date->days[$date->day]) {
    drupal_goto(_archive_url('all', $date->year, $date->month, $date->day));
  }
  else if ($date->months[$date->month]) {
    drupal_goto(_archive_url('all', $date->year, $date->month));
  }
  else if ($date->years[$date->year]) {
    drupal_goto(_archive_url('all', $date->year));
  }
  else if (arg(0) != 'archive') {
    drupal_goto('archive');
  }
  else {
    // Set the title for an empty archive page
    drupal_set_title(t('Archive'));    
    $output = '<p>'. t('There\'s currently nothing in the archives to browse.') .'</p>';
    return $output;
  }
}

function _get_month_names(){
  return array('', t('Jan'), t('Feb'), t('Mar'), t('Apr'), t('May'), t('Jun'), t('Jul'), t('Aug'), t('Sep'), t('Oct'), t('Nov'), t('Dec'));
}

/**
 * Builds an SQL query array (query and parameters) to use
 * to display an archive page for the specified date.
 *
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object obtained from _archive_date()
 * @return
 *   An array of (query, param_start, param_end)
 */
function _archive_query($type, $date) {
  // Confine the display interval to only one day
  if ($date->day) {
    $start = gmmktime(0, 0, 0, $date->month, $date->day, $date->year);
    $end   = gmmktime(0, 0, 0, $date->month, $date->day + 1, $date->year);
  }
  // Confine the display interval to one month
  else if ($date->month) {
    $start = gmmktime(0, 0, 0, $date->month, 1, $date->year);
    $end   = gmmktime(0, 0, 0, $date->month + 1, 1, $date->year);
  }
  // Confine the display interval to one year
  else if ($date->year) {
    $start = gmmktime(0, 0, 0, 1, 1, $date->year);
    $end   = gmmktime(0, 0, 0, 1, 1, $date->year + 1);
  }
  else {
    $start = 0;
    $end = 0;
  }
  
  // Grab limits on node types if exist
  $final_types = _archive_types_sql_string($type);
  
  // Allow viewing all nodes, not just nodes by year
  if ($start && $end) {
    return array('SELECT n.nid, n.type FROM {node} n WHERE n.status = 1 '. $final_types .'AND n.created >= %d AND n.created < %d ORDER BY n.created DESC', $start - $date->tz, $end - $date->tz);
  }
  else {
    return array('SELECT n.nid, n.type FROM {node} n WHERE n.status = 1 '. $final_types .'ORDER BY n.created DESC');
  }
}

/**
 * Returns the different node types that have nodes
 *
 * @param $date
 *    A date object obtained from _archive_date()
 * @return
 *    An array of node-types to number of posts of that type
 */
function _archive_node_types($date) {
  
  $types = variable_get('archive_type_filters', array());
  
  // Confine the display interval to only one day
  if ($date->day) {
    $start = gmmktime(0, 0, 0, $date->month, $date->day, $date->year);
    $end   = gmmktime(0, 0, 0, $date->month, $date->day + 1, $date->year);
  }
  // Confine the display interval to one month
  elseif ($date->month) {
    $start = gmmktime(0, 0, 0, $date->month, 1, $date->year);
    $end   = gmmktime(0, 0, 0, $date->month + 1, 1, $date->year);
  }
  // Confine the display interval to one year
  elseif ($date->year) {
    $start = gmmktime(0, 0, 0, 1, 1, $date->year);
    $end   = gmmktime(0, 0, 0, 1, 1, $date->year + 1);
  }
  else {
    $start = 0;
    $end = 0;
  }
  
  $result = '';
  if ($start && $end) {
    $result = db_query(db_rewrite_sql("SELECT t.type, t.name, COUNT(n.nid) AS node_count FROM {node} n INNER JOIN {node_type} t ON t.type = n.type WHERE n.status = 1 AND t.type IN ('". join($types, '", "') ."') AND n.created BETWEEN %d AND %d GROUP BY n.type ORDER BY n.created"), $start - $date->tz, $end - $date->tz);
  }
  else {
    $result = db_query(db_rewrite_sql("SELECT t.type, t.name, COUNT(n.nid) AS node_count FROM {node} n INNER JOIN {node_type} t ON t.type = n.type WHERE n.status = 1 AND t.type IN ('". join($types, "', '") ."') GROUP BY n.type ORDER BY n.created"));
  }
  
  $n_types = array();
  while ($row = db_fetch_array($result)) {
    $n_types[$row['type']] = array('count' => $row['node_count'],
                                   'name'  => $row['name']);
  }
  
  ksort($n_types);
  return $n_types;
}

function theme_archive_page_title($type, $year, $month, $day) {
  
  $title = t('Archive');
  $month_names = _get_month_names();
  if ($day) {
    $title .= ' - '. $month_names[$month] .' '. $day .', '. $year;
  }
  else if ($month) {
    $title .= ' - '. $month_names[$month] .' '. $year;
  }
  else if ($year) {
    $title .= ' - '. $year;
  }
  if ($type != 'all') {
    $type_name = db_result(db_query("SELECT name FROM {node_type} WHERE type = '%s'", $type));
    $title .= ' - '. ($type_name ? $type_name : $type);
  }
  
  return $title;
}


/**
 * Theme the archive navigation with years, months and dates by default.
 * 
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object returned from _archive_date()
 */
function theme_archive_navigation($type, $date) {
  $output  = "<div id=\"archive-container\"><dl><dt>". t('Date') ."</dt><dd>\n";
  $output .= theme('archive_navigation_years', $type, $date);
  if (_archive_validate_year($date->year)) {
    $output .= theme('archive_navigation_months', $type, $date);
  }
  if (_archive_validate_month($date->month)) {
    $output .= theme('archive_navigation_days', $type, $date);
  }
  $output .= "</dd>";
  // Only display node type filter if more than one node type represented
  if (count(_archive_node_types($date)) > 1) {
    $output .= "<dt>". t('Type') ."</dt><dd>\n";
    $output .= theme('archive_navigation_node_types', $type, $date);
    $output .= "</dd>";
  }
  $output .= "</dl></div>\n";
  return $output;
}

/**
 * Theme the list of years for the archive navigation.
 * 
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object returned from _archive_date()
 */
function theme_archive_navigation_years($type, $date) {
  $output = "<ul id=\"archive-years\">\n";
  $all_count = 0;
  foreach ($date->years as $year) {
    $all_count += $year;
  }
  $output .= '<li'. ($date->year?'':' class="selected"') .'>'. l(t('All'), _archive_url($type), array('title' => format_plural($all_count, '1 post', '@count posts'))) ."</li>\n";
  foreach ($date->years as $year => $post_count) {
    $class = '';
    if ($year == $date->year) {
      $class = ' class="selected"';
    }
    $output .= '<li'. $class .'>'. l($year, _archive_url($type, $year), array('title' => format_plural($post_count, '1 post', '@count posts'))) ."</li>\n";
  }
  $output .= "</ul>\n";

  return $output;
}

/**
 * Theme the list of months for the archive navigation.
 *
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object returned from _archive_date()
 */
function theme_archive_navigation_months($type, $date) {
  $output = "<ul id=\"archive-months\">\n";
  
  $all_count = 0;
  foreach ($date->months as $month) {
    $all_count += $month;
  }
  
  $output .= '<li'. ($date->month?'':' class="selected"') .'>'. l(t('All'), _archive_url($type, $date->year), array('title' => format_plural($all_count, '1 post', '@count posts'))) ."</li>\n";
  $curr_month = date('n', time());
  $curr_year = date('Y', time());
  $month_names = _get_month_names();
  foreach (range(1, 12) as $month) {
    $posts = !empty($date->months[$month]) ? $date->months[$month] : 0;
    $class = '';
    if ($month == $date->month) {
      $class = ' class="selected"';
    }
    elseif ($curr_year == $date->year && $month > $curr_month) {
      $class = ' class="future"';
    }
    $output .= "<li$class>". ($posts > 0 ? l($month_names[$month], _archive_url($type, $date->year, $month), array('title' => format_plural($posts, '1 post', '@count posts'))) : $month_names[$month]) ."</li>\n";
  }
  $output .= "</ul>\n";
  return $output;
}

/**
 * Theme the list of days for the archive navigation.
 * 
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object returned from _archive_date()
 */
function theme_archive_navigation_days($type, $date) {
  $output = "";
  $day_stop = gmdate('t', gmmktime(0, 0, 0, $date->month, 1, $date->year));
  $output = "<ul id=\"archive-days\">\n";
  
  $all_count = 0;
  foreach ($date->days as $day) {
    $all_count += $day;
  }
  
  $output .= '<li'. ($date->day?'':' class="selected"') .'>'. l(t('All'), _archive_url($type, $date->year, $date->month), array('title' => format_plural($all_count, '1 post', '@count posts'))) ."</li>\n";
  $curr_month = date('n', time());
  $curr_year = date('Y', time());
  $curr_day = date('j', time());
  for ($day = 1; $day <= $day_stop; $day++) {
    $posts = array_key_exists($day, $date->days) ? $date->days[$day] : 0;
    $class = '';
    if ($day == $date->day) {
      $class = ' class="selected"';
    }
    elseif ($curr_year == $date->year && $curr_month == $date->month && $day > $curr_day) {
      $class = ' class="future"';
    }
    $output .= "<li$class>". ($posts ? l($day, _archive_url($type, $date->year, $date->month, $day), array("title" => format_plural($posts, "1 post", "@count posts"))) : $day) ."</li>\n";
  }
  $output .= "</ul>\n";
  return $output;
}

/**
 * Theme the list of node types for the archives
 * 
 * @param $type
 *   A string representing the node-type currently being displayed
 * @param $date
 *   A date object returned from _archive_date()
 */
function theme_archive_navigation_node_types($type, $date) {
  $output = "<ul id=\"archive-node_types\">\n";
  $types_count = _archive_node_types($date);
  
  $all_count = 0;
  foreach ($types_count as $t) {
    $all_count += $t['count'];
  }
  
  $output .= '<li'. ($type != 'all'?'':' class="selected"') .'>'. l(t('All'), _archive_url('all'), array('title' => format_plural($all_count, '1 post', '@count posts'))) ."</li>\n";
  foreach ($types_count as $ft_key => $ft_value) {
    if (!$ft_value['count']) {
      continue;
    }
    $class = ($ft_key == $type ? ' class="selected"' : '');
    $name = $ft_value['name'];
    if ($types_count[$ft_key]['count'] > 0) {
      $output .= "<li$class>". l($name, _archive_url($ft_key, $date->year, $date->month, $date->day), array("title" => format_plural($types_count[$ft_key]['count'], "1 post", "@count posts"))) ."</li>\n";
    }
    else {
      $output .= "<li$class>$name</li>\n";
    }
  }
  $output .= "</ul>\n";
  return $output;
}
