import { LitElement, html } from 'lit';
import { customElement, property } from 'lit/decorators.js';
import { Chart, registerables } from 'chart.js';

// Register all Chart.js components
Chart.register(...registerables);

// Define a proper type for the chart data
interface ChartDataset {
  label?: string;
  data: number[];
  backgroundColor?: string | string[];
  borderColor?: string | string[];
  borderWidth?: number;
  fill?: boolean;
  tension?: number;
  [key: string]: any;
}

interface ChartData {
  labels: string[];
  datasets: ChartDataset[];
}

/**
 * A chart element web component using Chart.js.
 *
 * @prop chartType - The type of chart to display (bar, line, pie, etc.)
 * @prop chartData - The chart data in JSON format or as an object
 */
@customElement('chart-element')
export class ChartElement extends LitElement {
  /**
   * The type of chart (bar, line, pie, etc.)
   */
  @property({ type: String, attribute: 'chart-type' })
  chartType = 'bar';

  /**
   * The chart data - can be a JSON string or a direct object
   */
  @property({ type: Object, attribute: 'chart-data' })
  chartData: any = null;

  private chart: Chart | null = null;
  private canvasElement: HTMLCanvasElement | null = null;

  render() {
    return html`
      <div class="chart-container">
        <canvas id="chart"></canvas>
      </div>
    `;
  }

  // Important: Explicitly disable Shadow DOM for Chart.js to work properly
  createRenderRoot() {
    return this;
  }

  firstUpdated() {
    setTimeout(() => this.initChart(), 100);
  }

  connectedCallback() {
    super.connectedCallback();
  }

  updated(changedProperties: Map<string, any>) {
    if (changedProperties.has('chartType') || changedProperties.has('chartData')) {
      setTimeout(() => this.updateChart(), 100);
    }
  }

  private initChart() {
    const canvas = this.querySelector('#chart') as HTMLCanvasElement;
    if (!canvas) return;
    this.canvasElement = canvas;
    this.updateChart();
  }

  private updateChart() {
    if (!this.canvasElement) return;

    // Destroy existing chart if it exists
    if (this.chart) {
      this.chart.destroy();
    }

    let chartData: ChartData;

    // Parse data if needed
    try {
      // Handle string data (assuming it's valid JSON)
      if (typeof this.chartData === 'string') {
        chartData = JSON.parse(this.chartData as string);
      }
      // Handle object data directly
      else if (typeof this.chartData === 'object' && this.chartData !== null) {
        chartData = this.chartData as ChartData;
      }
      // Empty data as fallback
      else {
        chartData = { labels: [], datasets: [] };
      }

      // Create the chart with the parsed data
      this.chart = new Chart(this.canvasElement, {
        type: this.chartType as any,
        data: chartData,
        options: {
          responsive: true,
          maintainAspectRatio: false,
          animation: {
            duration: 1000
          }
        }
      });
    } catch (e) {
      // If any error occurs, create an empty chart
      this.chart = new Chart(this.canvasElement, {
        type: this.chartType as any,
        data: { labels: [], datasets: [] },
        options: {
          responsive: true,
          maintainAspectRatio: false
        }
      });
    }
  }

  disconnectedCallback() {
    super.disconnectedCallback();
    // Clean up chart when element is removed
    if (this.chart) {
      this.chart.destroy();
    }
  }
}

declare global {
  interface HTMLElementTagNameMap {
    'chart-element': ChartElement;
  }
}
