import { defineConfig } from 'vite';
import { resolve,relative } from 'path';
import { readdirSync, existsSync } from 'fs';
import tailwindcss from '@tailwindcss/vite';

// Changing server port implies changing the port in theme settings.
const serverPort = 3000;
const serverProtocol = 'https';

// Override with the domain of your site for specific origins.
// let serverOriginHost = 'https://my-drupal-site.ddev.site';
let serverOriginHost = null;

// Try to deduce server host from most known drupal local infrastructures.
if (serverOriginHost == null) {
  let serverHostEnvVarCandidates = [
    'DDEV_HOSTNAME',
    'VIRTUAL_HOST',
  ];

  for (let serverHostEnvVarIndex in serverHostEnvVarCandidates) {
    let serverHostEnvVar = serverHostEnvVarCandidates[serverHostEnvVarIndex];
    if (process.env[serverHostEnvVar] != null) {
      serverOriginHost = process.env[serverHostEnvVar];
      break;
    }
  }

}

const componentsDir = 'components';
let inputDeclaration = [];
let components = {};

// Components can be built using global or main vite build or isolated.
// Que diferencia hay entre añadir un fichero dentro o fuera de src?
readdirSync(componentsDir).forEach((dir) => {
  let paths = [
    resolve(__dirname, componentsDir, dir, 'src', `${dir}.ts`),
    resolve(__dirname, componentsDir, dir, 'src', `${dir}.js`),
    resolve(__dirname, componentsDir, dir, 'src', `${dir}.css`),
  ];
  paths.forEach((path) => {
    if (existsSync(path)) {
      inputDeclaration.push(path);
      components[dir] = dir;
    }
  });
});

export default defineConfig(({ mode }) => {
  const basePath = '/' + relative(__dirname.split('/themes/')[0], __dirname + '/dist/');
  return {
    base: basePath,
    publicDir: 'assets/',
    resolve: {
      alias: {
        '@src': resolve(__dirname, './src'),
        '@components': resolve(__dirname, './components'),
        '@daisyui': resolve(__dirname, './src/daisyui'),
        '@swiper': resolve(__dirname, './node_modules/swiper'),
        // '@artisan': resolve(__dirname.split('/themes/')[0], './themes/contrib/artisan')
        // '@vendor': resolve(__dirname, './node_modules'),
      }
    },
    server: {
      host: true,
      port: serverPort,
      allowedHosts: true,
      strictPort: true,
      cors: true,
      origin: serverProtocol + '://' + serverOriginHost + ':' + serverPort,
    },
    plugins: [tailwindcss()],
    build: {
      assetsInlineLimit: 0,
      manifest: true,
      outDir: 'dist',
      rollupOptions: {
        input: [
          resolve(__dirname, 'index.html'),
          resolve(__dirname, 'src/main.ts'),
          resolve(__dirname, 'src/main.css'),
          resolve(__dirname, 'src/shared-form.css'),
          ...inputDeclaration
        ],
        output: {
          entryFileNames: ({ name }) => {
            if (name in components) {
              return `${componentsDir}/${name}/[name].js`;
            }
            return '[name].js';
          },
          chunkFileNames: ({ name }) => {
            if (name in components) {
              return `${componentsDir}/${name}/[name].js`;
            }
            return '[name].js';
          },
          assetFileNames: ({ name, originalFileName }) => {
            if (name in components) {
              return `${componentsDir}/${name}/[name].[ext]`;
            }
            if (originalFileName && originalFileName.includes('components/') && originalFileName.split('components/')[1].split('/')[0] in components) {
              return `${componentsDir}/${originalFileName.split('components/')[1].split('/')[0]}/[name].[ext]`;
            }
            return '[name].[ext]';
          },
          manualChunks(id) {
            if (id.includes('node_modules')) {
              return 'vendor/' + id.toString().split('node_modules/')[1].toString().replace('/', '_').split('/')[0].toString().replace('.js', '');
            }
          }
        }
      },
      emptyOutDir: true
    }
  };
});
