<?php

namespace Drupal\artisan_styleguide\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Theme\ComponentPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns responses for Artisan styleguide routes.
 */
final class ArtisanStyleguideController extends ControllerBase {

  /**
   * The controller constructor.
   */
  public function __construct(
    private readonly RendererInterface $renderer,
    private readonly ComponentPluginManager $pluginManagerSdc,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): self {
    return new self(
      $container->get('renderer'),
      $container->get('plugin.manager.sdc'),
    );
  }

  /**
   * Builds main Artisan styleguide.
   */
  public function __invoke(): array {
    $components = [];
    $build = [];
    foreach ($this->pluginManagerSdc->getDefinitions() as $plugin_id => $definition) {
      $components[] = $this->componentDefinitionPreview($plugin_id, $definition);
    }
    $build['content'] = [
      '#theme' => 'artisan_styleguide',
      '#cache' => ['max-age' => 0],
      '#intro_notes' => [
        $this->t('Note customizations of main theme uses CSS variables like "--theme-palette-primary". Make sure you take that into account in order to preserve site design harmony.'),
        $this->t('Theme customization usage format "--example-color: var(--theme-example-color, var(--FALLBACK-CSS-VAR, FALLBACK-CSS-VALUE));".'),
        $this->t('Theme customization usage defined format "color: var(--example-color);".'),
        $this->t('Ensure correct prop (predefined schema simple value) or slot (renderable "block") definitions for components.'),
      ],
      '#components' => $components,
    ];

    return $build;
  }

  /**
   * Component definition preview.
   *
   * @param string $plugin_id
   *   Plugin id e.g "artisan:header".
   * @param array $definition
   *   Plugin definition.
   *
   * @return array
   *   Renderable component preview.
   */
  protected function componentDefinitionPreview($plugin_id, array $definition) {
    $component = [
      '#type' => 'component',
      '#component' => $plugin_id,
      '#props' => [],
      '#slots' => [],
    ];
    $clarifications = [];
    $name = $definition['name'] ?? $this->t('TBD');
    $empty_props_slots = TRUE;
    self::componentDefinitionPreviewProcessProps($definition, $component, $clarifications, $empty_props_slots);
    self::componentDefinitionPreviewProcessSlots($definition, $component, $clarifications, $empty_props_slots);
    // Uncomment for debug:
    // @code if ($plugin_id == 'artisan_styleguide:artisan-styleguide-sdc-model') { @endcode
    // @code   dump($component); @endcode
    // @code   dump($definition); @endcode
    // @code } @endcode
    try {
      $to_render = $component;
      $rendered = $this->renderer->renderPlain($to_render);
      $status = TRUE;
    }
    catch (\Exception $ex) {
      $rendered = '';
      $status = FALSE;
      $clarifications[] = $this->t('Unable to preview, ensure props & slots definitions, slots should be always render arrays or twig string (scalar): @error.', [
        '@error' => $ex->getMessage(),
      ]);
      $clarifications[] = $this->t('Note no attached CSS/JS in preview for this component due failed to render.');
    }
    if ($empty_props_slots) {
      $status = FALSE;
      $clarifications[] = $this->t('Missing props and/or slots definitions.');
    }
    if (empty(trim(strip_tags((string) $rendered)))) {
      $status = FALSE;
      $clarifications[] = $this->t('Empty output detected, please review definition & examples definitions.');
    }
    if ($status) {
      $clarifications[] = $this->t('It should look great!');
    }
    return [
      '#theme' => 'artisan_styleguide__component',
      '#status' => $status,
      '#plugin_id' => $plugin_id,
      '#name' => $name,
      '#rendered' => $rendered,
      '#component' => $component,
      '#clarifications' => $clarifications,
    ];
  }

  /**
   * Component definition preview process props.
   *
   * @param array $definition
   *   Component definition to process.
   * @param array $component
   *   Component result to process.
   * @param array $clarifications
   *   Clarifications.
   * @param bool $empty_props_slots
   *   Empty props or slots indicator.
   */
  protected function componentDefinitionPreviewProcessProps($definition, &$component, &$clarifications, &$empty_props_slots) {
    foreach ($definition['props']['properties'] ?? [] as $prop_key => $prop_def) {
      if (empty($prop_def['examples'] ?? NULL)) {
        $clarifications[] = $this->t('@prop - Missing examples for property declaration.', [
          '@prop' => $prop_key,
        ]);
        continue;
      }
      $empty_props_slots = FALSE;
      if ($prop_def['type'] && class_exists($prop_def['type'])) {
        $class = $prop_def['type'];
        $component['#props'][$prop_key] = new $class(reset($prop_def['examples']));
      }
      else {
        $component['#props'][$prop_key] = reset($prop_def['examples']);
      }
    }
  }

  /**
   * Component definition preview process slots.
   *
   * @param array $definition
   *   Component definition to process.
   * @param array $component
   *   Component result to process.
   * @param array $clarifications
   *   Clarifications.
   * @param bool $empty_props_slots
   *   Empty props or slots indicator.
   */
  protected function componentDefinitionPreviewProcessSlots(array $definition, array &$component, array &$clarifications, bool &$empty_props_slots) {
    foreach ($definition['slots'] ?? [] as $slot_key => $slot_def) {
      if (empty($slot_def['examples'] ?? NULL)) {
        $clarifications[] = $this->t('@slot - Missing examples for slot declaration.', [
          '@slot' => $slot_key,
        ]);
        continue;
      }
      $empty_props_slots = FALSE;
      $first_example = reset($slot_def['examples']);
      if (is_string($first_example)) {
        $first_example = [
          '#type' => 'inline_template',
          '#template' => $first_example,
        ];
      }
      $component['#slots'][$slot_key] = $first_example;
    }
  }

}
