<?php

namespace Drupal\asciidoc_display;

class AsciidocItem {

  /**
   * Array of items found in the asciidoc HTML document.
   *
   * @var array
   */
  private $values;

  /**
   * Options used by the parser when creating this item.
   *
   * @var array
   */
  private $configuration;

  /**
   * AsciidocItem constructor.
   */
  public function __construct(array $values, array $configuration) {
    $this->values = $values;
    $this->configuration = $configuration;
  }

  /**
   * Get the value of a configuration option used by the parser.
   *
   * See \Drupal\asciidoc_display\AsciidocParserInterface::parseBareHtml for a
   * list of options.
   *
   * @param string $key
   *   Name of the configuration option to retrieve.
   *
   * @return false|mixed
   */
  public function getConfiguration(string $key) {
    return $this->configuration[$key] ?? FALSE;
  }

  /**
   * Getter for book title.
   *
   * @return string|null
   *   The title of the book this page is in.
   */
  public function getBookTitle() {
    return $this->values['book_title'] ?? NULL;
  }

  /**
   * Getter for section title.
   *
   * @return string|null
   *   The title of this particular topic/section/page.
   */
  public function getSectionTitle() {
    return $this->values['section_title'] ?? NULL;
  }

  /**
   * Getter for body.
   *
   * @return string|null
   *   The HTML of the page.
   */
  public function getBody() {
    return $this->values['body'] ?? NULL;
  }

  /**
   * Getter for summary.
   *
   * @return string|null
   *   The HTML summary of the page if it was set.
   */
  public function getSummary() {
    return $this->values['summary'] ?? NULL;
  }

  /**
   * Getter for table of contents.
   *
   * @return string|null
   *   The HTML for the table of contents section, if any.
   */
  public function getToc() {
    return $this->values['toc'] ?? NULL;
  }

  /**
   * Getter for parents.
   *
   * @return array|null
   *   The parents array for the table of contents section. This is an
   *   associative array of file name => info, and info has keys:
   *   - parent: Parent file name.
   *   - weight: Order of the item in the list.
   */
  public function getParents() {
    return $this->values['parents'] ?? NULL;
  }

  /**
   * Get item source files.
   *
   * @return array|null
   *   An array of source files for the page, if any are noted. Validated to
   *   contain only legal characters.
   */
  public function getSourceFiles() {
    return $this->values['files'] ?? NULL;
  }

  /**
   * Get item copyright.
   *
   * @return string|null
   *   The HTML copright notice for the page, if any.
   */
  public function getCopyright() {
    return $this->values['copyright'] ?? NULL;
  }

  /**
   * Get item images.
   *
   * @return array|null
   *   An array of 'src' attributes for images found in the page.
   */
  public function getImages() {
    return $this->values['images'] ?? NULL;
  }

  /**
   * Get item type.
   *
   * @retrun string|null
   *   One of 'book', 'chapter', or 'topic'.
   */
  public function getType() {
    return $this->values['type'] ?? NULL;
  }

}
