<?php

namespace Drupal\asciidoc_display\Form;

use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;

class EditForm extends FormBase {

  /**
   * File system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * EditForm constructor.
   *
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   File system service.
   */
  public function __construct(FileSystemInterface $file_system) {
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritDoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('file_system')
    );
  }

  /**
   * {@inheritDoc}
   */
  public function getFormId() {
    return 'asciidoc_edit_form';
  }

  /**
   * Form builder for the AsciiDoc editing form.
   *
   * @param array $form
   *   The Form API array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state.
   * @parm string $file
   *   File name being edited.
   * @param string $source_dir
   *   The system path to the source directory.
   * @param bool $translated
   *   TRUE if the book has multiple languages, in which case the files are
   *   assumed to be in $source_dir within language directories. FALSE if not,
   *   in which case the files are directly in $source_dir.
   * @param string $langcode
   *   Language code, if there is one.
   * @param string $source_langcode
   *   The language code of the source language for the book.
   * @param string $path_prefix
   *   Path prefix for editing links for other langauges. The links will be made
   *   to "$path_prefix/$langcode/$file" if the book is translated.
   * @param bool $do_patches
   *   TRUE if submitting the form should download a patch file; FALSE if it
   *   should download the edited source file.
   * @param string $line_endings
   *   Type of line endings to use: 'unix' for Unix/Linux; anything else for
   *   Windows.
   * @param string $image_url
   *   URL for image directory.
   */
  public function buildForm(array $form, FormStateInterface $form_state, $file = NULL, $source_dir = NULL, $translated = NULL, $langcode = NULL, $source_langcode = NULL, $path_prefix = NULL, $do_patches = NULL, $line_endings = NULL, $image_url = NULL) {
    $form['#asciidoc_source_file'] = $file;
    $form['#asciidoc_do_patches'] = $do_patches;
    $form['#asciidoc_line_endings'] = $line_endings;
    $form['#asciidoc_source_dir'] = $source_dir;

    $text = _asciidoc_display_load_source($file, $source_dir, $translated, $langcode);
    if (!$text) {
      $form['error'] = [
        '#type' => 'markup',
        '#markup' => t('Could not read %file', ['%file' => $file]),
      ];
      return $form;
    }

    if ($translated) {
      $other = asciidoc_display_edit_other_languages($file, $source_dir, $langcode, $source_langcode, $path_prefix);
      if ($other) {
        $form['other_language'] = $other;
      }
    }

    $form['help'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('You are editing the source for a page that is formatted using the <a href="http://asciidoc.org">AsciiDoc markdown format</a>. After editing, download your revised source and attach it to an issue.') . '</p><p>' . t('There is a Preview button in the toolbar. It will show you the formatting output, except that cross links do not show up with topic titles as the link text.') . '</p>',
    ];

    $form['editor'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Edit %file', ['%file' => $file]),
      '#default_value' => $text,
      '#rows' => 20,
      '#cols' => 80,
      '#attributes' => ['class' => ['mark-it-up']],
    ];

    $form['#attached']['library'][] = [
      'asciidoc_display/edit',
    ];
    $form['#attached']['js'][] = [
      'data' => [
        'asciidoc_display' => [
          'image_dir' => $image_url,
        ],
      ],
      'type' => 'setting',
    ];

    if ($do_patches) {
      $label = $this->t('Download patch file');
    }
    else {
      $label = $this->t('Download new file');
    }

    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $label,
    ];

    return $form;
  }

  /**
   * Form submit handler.
   *
   * Either display the edited text. Or generate .patch file and send that to
   * the browser.
   *
   * @param array $form
   *   The Form API array.
   * @param Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $file = $form['#asciidoc_source_file'];
    $text = $form_state['values']['editor'];
    $line_endings = $form['#asciidoc_line_endings'];
    $source_dir = $form['#asciidoc_source_dir'];
    $do_patches = $form['#asciidoc_do_patches'];

    // The text editor prefers DOS line endings. Convert back to Unix, if
    // requested.
    if ($line_endings == 'unix') {
      $text = str_replace("\r", '', $text);
    }

    if (!$text) {
      $this->messenger()->addMessage($this->t('No text in editor'));
      return;
    }

    // Make a patch file, optionally.
    if ($do_patches) {
      $path = \Drupal::service("file_system")->realpath($source_dir . '/' . $file);
      $old_text = @file_get_contents($path);
      if ($old_text) {
        $text = xdiff_string_diff($old_text, $text);
      }

      // If we made a diff, change the filename.
      $file = $file . '.patch';
    }

    // Send the patch or text file to the browser.
    $response = new Response();
    $response->headers->set('Content-Type', 'text/plain');
    $response->headers->set('Content-Disposition', sprintf('attachment; filename="%s"', $file));
    $response->setContent($text);
    return $response;
  }

}
