<?php
// $Id: AssistantContext.inc,v 1.1 2010/03/09 16:54:51 pounard Exp $

/**
 * @file
 * Solace API Context OOP abstraction.
 */

class AssistantContextException extends Exception { }

/**
 * Default interface for all contextes.
 * 
 * Note for developers, in order to create your own Context implementation,
 * inherit from this class and override any methods, sticking to the API
 * documentation.
 * 
 * @see assistant_api_context_create()
 */
abstract class AbstractAssistantContext
{
  /**
   * Filters instance id.
   * 
   * @var integer
   */
  protected $_fid = NULL;

  /**
   * Return the current filters instance id.
   * 
   * @return int
   *   Filters instance id
   */
  public function getFid() {
    return $this->_fid;
  }

  /**
   * Does this context allow save button in filter form.
   * 
   * @var boolean
   */
  protected $_save = FALSE;

  /**
   * Get owner module name.
   * 
   * @return string
   *   Internal module name that owns this context
   */
  public abstract function getModule();

  /**
   * Does this context allow previews in filter form.
   * 
   * @return boolean
   *   TRUE if the context allow previews.
   */
  public abstract function allowPreview();

  /**
   * Current context execution flow is in preview mode.
   * 
   * @var boolean
   */
  private $__preview = FALSE;

  /**
   * Are we currently in preview mode?
   * 
   * @return boolean
   */
  public function isPreview() {
    return $this->__preview;
  }

  /**
   * Set the context execution flow in preview mode. This can be usefull to tell
   * hook_assistant_api_query_alter() implementations that the SolR query being
   * built is meant for preview.
   * 
   * @param boolean $toggle
   *   TRUE to set in preview mode, FALSE else
   * 
   * @throws AssistantContextException
   *   If context does not supports preview mode.
   */
  public function setPreview($toggle) {
    if (!$this->allowPreview()) {
      $this->__preview = FALSE;
      throw new AssistantContextException("This context does not supports preview");
    }
    $this->__preview = (bool) $toggle;
  }

  /**
   * Keep a track of last generated query.
   * 
   * @var array
   */
  private $__lastQuery = NULL;

  /**
   * Return last done query
   * 
   * @return array
   *   An array filled with q, fq, and other query parameters
   *   NULL if this context has not been used to build a query
   */
  public function getLastPlainQuery() {
    return $this->__lastQuery;
  }

  /**
   * Return last done query
   * 
   * @param array $params
   *   An array filled with q, fq, and other query parameters
   */
  public function setLastPlainQuery(array $params) {
    $this->__lastQuery = $params;
  }
  
  /**
   * Return the valid token types for current context. This will be used by
   * token into some filters.
   * 
   * @return array
   *   Array of token types
   */
  public function getAllowedTokenTypes() {
    // Default context should not allow any other values that those.
    return array('global');
  }

  /**
   * Get object for given token type. Any custom implementation that provides
   * an getAllowedTokenTypes() override better should implement this method in
   * order to supports their own custom tokens. 
   * 
   * @return mixed
   *   Any object. NULL if no object is required for given type.
   */
  protected function _getTokenObjectForType($type) {
    // This is a sample implementation, switch here is useless but remains a
    // good code sample for custom implementations.
    switch ($type) {
      case 'global':
      default:
        return NULL;
    }
  }

  /**
   * Proceed token replacements in given text. This is the only external entry
   * point for token feature, so this is where the token_module existance will
   * be done.
   * 
   * @param string $text
   * 
   * @return string
   *   Altered $text
   */
  public final function tokenReplace($text) {
    if (module_exists('token')) {
      foreach ($this->getAllowedTokenTypes() as $type) {
        $text = token_replace($text, $type, $this->_getTokenObjectForType($type));
      }
    }
    return $text;
  }

  /**
   * Help about your token target objets. We highly recommend any overriding
   * class to override this method if they provide extra token types.
   * 
   * @return string
   *   Human readable and localized string
   */
  public function getTokenHelp() {
    return t('All the tokens are relative to global hosting site.');
  }

  /**
   * Does this context allow save button in filter form.
   * 
   * @return boolean
   *   TRUE if the context allow save button.
   */
  public function allowSave() {
    return $this->_save;
  }

  /**
   * Default constructor.
   * 
   * @param $fid
   *   Filters instance id
   * @param boolean $save = FALSE
   *   If set to TRUE, this will alter the form behavior and add the filters save
   *   submit function, see assistant_api_filters_form() documentation
   */
  public function __construct($fid = NULL, $save = FALSE) {
    if ($fid) {
      $this->_fid = $fid;
    }
    $this->_save = (bool) $save;
    if (! $fid && $this->_save) {
      // TODO change this when filters will have a full OOP API.
      $this->_fid = assistant_api_filters_empty_instance();
    }
  }
}

/**
 * Default implementation for simple modules that don't intend to override
 * contextes.
 */
class DefaultAssistantContext extends AbstractAssistantContext
{
  /**
   * Module name.
   * 
   * @var string
   */
  private $__module = 'global';

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantContext#getModule()
   */
  public function getModule() {
    return $this->__module;
  }

  /**
   * Allow usage of this implementation for any module that does not wishes to
   * override the AbstractAssistantContext class.
   * 
   * @param string $module
   *   Internal module name.
   */
  public function setModule($module) {
    $this->__module = $module;
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantContext#allowPreview()
   */
  public function allowPreview() {
    return FALSE;
  }
}

/**
 * Context implementation that ensures that the given context is build arround
 * a particular node.
 */
class NodeTypeAssistantContext extends AbstractAssistantContext
{
  /**
   * (non-PHPdoc)
   * @see AbstractAssistantContext#getModule()
   */
  public function getModule() {
    return 'node';
  }
  
  /**
   * Current node type.
   * 
   * @var string
   */
  private $__nodeType = NULL;

  /**
   * Set node nid for this context.
   * 
   * @param string $node_type
   *   Node type
   */
  public function setNodeType($nodeType) {
    $this->__nodeType = $nodeType;
  }

  /**
   * Get current node nid.
   * 
   * @return integer
   *   Current node nid
   * 
   * @throws AssistantContextException
   *   If no node nid set
   */
  public function getNodeType() {
    if (! $this->__nodeType) {
      throw new AssistantContextException("No node type set");
    }
    return $this->__nodeType;
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantContext#allowPreview()
   */
  public function allowPreview() {
    return FALSE;
  }
}

/**
 * Context implementation that ensures that the given context is build arround
 * a particular node.
 */
class NodeAssistantContext extends NodeTypeAssistantContext
{
  /**
   * Current node nid.
   * 
   * @var integer
   */
  private $__nid = NULL;

  /**
   * Set node nid for this context.
   * 
   * @param integer $nid
   *   Node nid
   */
  public function setNid($nid) {
    $this->__nid = $nid;
  }

  /**
   * Get current node nid.
   * 
   * @return integer
   *   Current node nid
   * 
   * @throws AssistantContextException
   *   If no node nid set
   */
  public function getNid() {
    if (! $this->__nid) {
      throw new AssistantContextException("No nid set");
    }
    return $this->__nid;
  }

  /**
   * Get current node
   * 
   * @return object
   *   Fully loaded node object
   * 
   * @throws AssistantContextException
   *   If no node nid set
   */
  public function getNode() {
    return node_load($this->getNid());
  }

  /**
   * (non-PHPdoc)
   * @see NodeTypeAssistantContext#getNodeType()
   */
  public function getNodeType() {
    try {
      return parent::getNodeType();
    }
    catch (AssistantContextException $e) {
      $node = $this->getNode();
      parent::setNodeType($node->type);
      return $node->type;
    }
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantContext#getAllowedTokenTypes()
   */
  public function getAllowedTokenTypes() {
    return array('global', 'node');
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantContext#_getTokenObjectForType($type)
   */
  protected function _getTokenObjectForType($type) {
    switch ($type) {
      case 'node':
        return node_load($this->getNid());
      case 'global':
      default:
        return NULL;
    }
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantContext#getTokenHelp()
   */
  public function getTokenHelp() {
    return t("Tokens relative to 'node' are information about the current assisted node.");
  }
}
