<?php
// $Id: AssistantFilter.inc,v 1.1 2010/03/09 16:54:51 pounard Exp $

/**
 * @file
 * Solace API Filter OOP abstraction.
 */

/**
 * Generic execption for any errors thrown by the Assistant Filters API
 */
class AssistantFilterException extends Exception { }

/**
 * Exception thrown by AssistantFilterFactory.
 */
class AssistantFilterFactoryException extends AssistantFilterException { }

/**
 * Factory class that allows to bypass Drupal API for some common operations.
 * Note that it must be initialized before every usage.
 * 
 * @see assistant_api_filters_boostrap()
 */
class AssistantFilterFactory
{
  /**
   * Static cache of internal names for rapid access.
   * 
   * @var array
   */
  private static $__names = NULL;

  /**
   * Static cache of custom module given filter descriptions.
   * 
   * @var array
   */
  private static $__cache = NULL;

  /**
   * Lazzy loaded instances static cache.
   * 
   * @var array
   */
  private static $__instances = array(); 

  /**
   * Get all supported filter names.
   * 
   * @return array
   *   Array of internal filter name.
   */
  public static function getFilterNamesAll() {
    return self::$__name;
  }

  /**
   * Static initialization. We'll keep this one lazzy to ensure we won't do any
   * unneeded operations.
   * 
   * Drupal side will have to initialize this object using the cache.
   * 
   * @param array $cache
   *   Key/value pairs. Keys are filter names, values are descriptive array
   *   resulting of cache construction, with invalid cuustom filter declarations
   *   excluded.
   */
  public static function init(&$cache) {
    self::$__cache = &$cache;
    self::$__names = array_keys(self::$__cache);
  }

  /**
   * Get specific filter instance by name.
   * 
   * @param string $name
   *   Filter name
   * @param AbstractAssistantContext $context = NULL
   *   (optional) AbstractAssistantContext instance on which the filter should
   *   react.
   * 
   * @return AbstractAssistantFilter
   *   AbstractAssistantFilter specific implementation.
   *   NULL if filter does not exists.
   * 
   * @throws AssistantFilterFactoryException
   *   If filter name does not exists
   */
  public static function getFilterInstanceByName($name) {
    // Check filter name is registered
    if (!isset(self::$__cache[$name])) {
      throw new AssistantFilterFactoryException("Unknown filter name: " . (string) $name);
    }

    if (! isset(self::$__instances[$name])) {
      // Proceed to lazzy instanciation.
      $desc = &self::$__cache[$name];
      if (isset($desc['file'])) {
        require_once $desc['file'];
      }
      self::$__instances[$name] = new $desc['class']($name);
    }

    return self::$__instances[$name];
  }

  /**
   * Get specific filters instances available under the given context.
   * 
   * @return array
   *   Array of AbstractAssistantFilter instances
   */
  public static function getFilterNamesByContext(AbstractAssistantContext $context) {
    $ret = array();

    // This could be a lot faster and avoid so much objets instanciation.
    foreach (self::$__cache as $name => &$desc) {
      $filter = self::getFilterInstanceByName($name);
      $contextes = $filter->getAllowedContextClassNames();
      if (empty($contextes)) {
        $ret[] = $name;
      }
      else {
        foreach ($contextes as $className) {
          if (get_class($context) == $className || is_subclass_of($context, $className)) {
            $ret[] = $name;
            break;
          }
        }
      }
    }

    return $ret;
  }
}

/**
 * Default interface for all filters.
 * 
 * This class must be overriden in order to properly make a filter for the
 * system. Remember that *any* methods which is not final can (and in most
 * cases should) be overriden.
 */
abstract class AbstractAssistantFilter
{
  /**
   * Tells if current filter is boost-able.
   */
  public function isBoostAble() {
    return FALSE;
  }

  /**
   * Tells if filter is fuzzy-able
   */
  public function isFuzzyAble() {
    return FALSE;
  }

  /**
   * Tells if filter is roaming-able
   */
  public function isRoamingAble() {
    return FALSE;
  }

  /**
   * Tells if the filter will use the filter query in order to do a really
   * restrictive query.
   * 
   * Set this flag is optional, but it will alter the final UI and tell the
   * user this flag is a query killer.
   */
  public function isFilterQuery() {
    return FALSE;
  }

  /**
   * Process values in order to do token replacements in all string ones.
   * 
   * @param AbstractAssistantContext $context
   *   AbstractAssistantContext instance.
   * @param array &$values
   *   Values from the form filter element.
   * 
   * @return unknown_type
   */
  public final function processValues(AbstractAssistantContext $context, &$values) {
    if (!empty($values)) {
      foreach ($values as $key => $value) {
        if (is_string($value) && strlen($value) > 0) {
          $values[$key] = $context->tokenReplace($value);
        }
      }
    }
  }

  /**
   * This is fired when the Solr request is built.
   * 
   * @param AbstractAssistantContext $context
   *   AbstractAssistantContext instance.
   * @param array &$values
   *   Values from the form filter element. This will also contain the 'boost' and
   *   'fuzzyness' parameters.
   * @param SolrQuery $query
   *   An SolrQuery instance object, ready to use
   * 
   * @return void
   *   No return value, here, you have to alter the $query object.
   */
  public final function build(AbstractAssistantContext $context, &$values, SolrQuery $query) {
    $this->processValues($context, $values);
    $this->_build($context, $values['values'], $query);
  }

  /**
   * Build single filter subform
   * 
   * @param AbstractAssistantContext $context
   *   AbstractAssistantContext instance.
   * @param array &$values = array()
   *   (optional) User set values in case of update
   * 
   * @return array
   *   Valid Form API elements subset
   */
  public final function form(AbstractAssistantContext $context, &$values = array()) {
    // Retrieve specific form
    $form = array();
    $form['title'] = array('#markup' => 'value', '#value' => t($this->getTitle()), '#weight' => -50);
    $form['values'] = $this->_form($context, $values['values']);
    $form['values']['#prefix'] = '<div id="' . $this->getValuesDivId() . '" class="filter-values-' . $this->getName() . '">';
    $form['values']['#suffix'] = '</div>';
    // Add fuzzyness slider if fuzzable
    /* TODO deactivated fuzzyness support
     * @see theme_assistant_api_filters_form_data() PHPdoc. 
    if ($filter['fuzzable']) {
      $form['fuzzy'] = array(
        '#type' => 'assistant_slider',
        '#title' => t('Fuzzy/Roaming factor'),
        '#default_value' => isset($values['fuzzy']) ? $values['fuzzy'] : 0,
      );
    }
    */
    // Add boost slider if boostable
    if ($this->isBoostAble()) {
      $form['boost'] = array(
        '#type' => 'assistant_slider',
        '#title' => t('Boost factor'),
        '#default_value' => isset($values['boost']) ? $values['boost'] : 0,
      );
    }
    // This will help for theming later
    $form['filter_query'] = array(
      '#type' => 'value',
      '#value' => $this->isFilterQuery(),
    );
    return $form;
  }

  /**
   * Validate form elements.
   * 
   * @param AbstractAssistantContext $context
   *   AbstractAssistantContext instance.
   * @param array &$values = array()
   *   (optional) User set values in case of update
   * 
   * @return array
   *   Array filled with two values, first one is the full Form API element path,
   *   relative to this subform root (it could be something like 'foo][bar][baz').
   *   Second value is the localized error message.
   *   Return NULL if no error happens.
   */
  public function validate(AbstractAssistantContext $context, &$values) {
    return $this->_validate($context, $values['values']);
  }

  /**
   * Returns a list of allowed context class for this filter.
   * 
   * @return array
   *   Array of context canonical class names, or NULL if filter can be used
   *   within all contextes.
   */
  public function getAllowedContextClassNames() {
    return NULL;
  }

  /**
   * Get filter title.
   * 
   * @return string
   *   Filter title.
   */
  public abstract function getTitle();

  /**
   * Get filter description.
   * 
   * @return string
   *   Filter description or NULL if not set.
   */
  public abstract function getDescription();

  /**
   * This is more a callback than a hook.
   * It is fired when the Solr request is built.
   * 
   * <code>
   *   protected function _build(AbstractAssistantContext $context, &$values, SolrQuery $query) {
   *     if (! empty($values['keywords'])) {
   *       $query->q->add($values['keywords'], NULL, $values['boost']);
   *     }
   *   }
   * </code>
   * 
   * @param AbstractAssistantContext $context
   *   AbstractAssistantContext instance.
   * @param array &$values
   *   Values from the form filter element. This will also contain the 'boost' and
   *   'fuzzyness' parameters.
   * @param SolrQuery $query
   *   An SolrQuery instance object, ready to use
   * 
   * @return void
   *   No return value, here, you have to alter the $query object.
   */
  protected abstract function _build(AbstractAssistantContext $context, &$values, SolrQuery $query);
  
  /**
   * This is more a callback than a hook.
   * It is fired when the filter form is built.
   * 
   * <code>
   *   public function form(AbstractAssistantContext $context, &$values = array()) {
   *     $form = array();
   *     $form['keywords'] = array(
   *       '#type' => 'textfield',
   *       '#default_value' => $values['keywords'],
   *       '#required' => TRUE,
   *     );
   *     return $form;
   *   }
   * </code>
   *
   * @param AbstractAssistantContext $context
   *   AbstractAssistantContext instance.
   * @param array &$values = array()
   *   Values from the form filter element, it can be empty in case of new element
   *   spawn in the filter form.
   *
   * @return array
   *   A Form API valid subpart of form.
   */
  protected abstract function _form(AbstractAssistantContext $context, &$values = array());
  
  /**
   * This is more a callback than a hook.
   * It is fired when the filter form is built.
   * 
   * <code>
   *   public function validate(AbstractAssistantContext $context, &$values = array()) {
   *     // This is foo code and will always fail
   *     return array('keywords', t('Some error'));
   *   }
   * </code>
   * 
   * @param AbstractAssistantContext $context
   *   AbstractAssistantContext instance.
   * @param array &$values
   *   Values from the form filter element.
   *
   * @return array
   *   Array filled with two values, first one is the full Form API element path,
   *   relative to this subform root (it could be something like 'foo][bar][baz').
   *   Second value is the localized error message.
   *   Return NULL if no error happens.
   */
  protected function _validate(AbstractAssistantContext $context, &$values) { }

  /**
   * Ahah helper path of current element.
   * 
   * @var array
   */
  private $__ahahHelperPath = NULL;

  /**
   * Set Ahah helper path of current element.
   * 
   * @param array $path
   */
  public final function setAhahHelperPath($path) {
    $this->__ahahHelperPath = $path;
    $this->__ahahHelperPath[] = 'values';
  }

  /**
   * Get Ahah helper path of current element.
   * 
   * @return array
   */
  protected final function _getAhahHelperPath() {
    return $this->__ahahHelperPath;
  }

  /**
   * Get computed div id using ahah_helper path.
   * 
   * @return string
   *   
   * @throws AssistantFilterException
   *   If ahah helper path is not set
   */
  public final function getValuesDivId() {
    if (!$this->__ahahHelperPath) {
      throw new AssistantFilterException("No Ahah helper path set");
    }
    return str_replace('_', '-', implode('-', $this->__ahahHelperPath));
  }

  /**
   * Set ahah property of current element. This will allow subfilters to use
   * AHAH. Note that because of form complexity, you can only set the changing
   * event, and take care yourself about what to do when the _form() method is
   * called using values.
   * 
   * Note that values are always stored into form storage, so what you get at
   * form build time is the current state of form, even in AJAX context (with
   * values filled).
   * 
   * @param array $element
   *   The form element on which you want to apply the #ahah property.
   * @param string $event = 'change'
   *   (optional) The event on which to react.
   */
  public final function setAhahProperty(&$element, $event = 'change') {
    $element['#ahah'] = array(
      'event' => 'click',
      'path' => ahah_helper_path($this->_getAhahHelperPath()),
      'wrapper' => $this->getValuesDivId(),
      'effect' => 'none',
      'method' => 'replace',
    );
  }

  /**
   * Filter internal name.
   * 
   * @var string
   */
  private $_name = NULL;

  /**
   * Get filter internal name.
   * 
   * @return string
   *   Filter internal name.
   */
  public final function getName() {
    return $this->_name;
  }

  /**
   * Constructor.
   * 
   * @param string $name
   *   Internal filter name.
   */
  public final function __construct($name) {
    $this->_name = (string) $name;
  }
}

class KeywordAssistantFilter extends AbstractAssistantFilter
{
  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#getTitle()
   */
  public function getTitle() {
    return "Common search query";
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#getDescription()
   */
  public function getDescription() {
    return "Like any search engine, this field is used as a global keyword search";
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#_build($context, $values, $query)
   */
  protected function _build(AbstractAssistantContext $context, &$values, SolrQuery $query) {
    if (! empty($values['keywords'])) {
      foreach (preg_split('/[\.-_, ]+/', $values['keywords']) as $word) {
        $query->q->add(new SolrTerm($word));
      }
    }
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#_form($context, $values)
   */
  protected function _form(AbstractAssistantContext $context, &$values = array()) {
    $form = array();
    $form['keywords'] = array(
      '#type' => 'textfield',
      '#default_value' => $values['keywords'],
      '#required' => TRUE,
    );
    return $form;
  }
}

class PhraseAssistantFilter extends AbstractAssistantFilter
{
  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#getTitle()
   */
  public function getTitle() {
    return "Exact phrase/word match";
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#getDescription()
   */
  public function getDescription() {
    return "Boostable keywords based lookup";
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#isBoostAble()
   */
  public function isBoostAble() {
    return TRUE;
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#_build($context, $values, $query)
   */
  protected function _build(AbstractAssistantContext $context, &$values, SolrQuery $query) {
    if (! empty($values['phrase'])) {
      $term = new SolrTerm($values['phrase']);
      $term->setBoost($values['boost']);
      // $term->setFuzzyness($values['fuzzy']);
      $query->q->add($term);
    }
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#_form($context, $values)
   */
  protected function _form(AbstractAssistantContext $context, &$values = array()) {
    $form = array();
    $form['phrase'] = array(
      '#type' => 'textfield',
      '#default_value' => $values['phrase'],
      '#required' => TRUE,
    );
    return $form;
  }
}

class NodeTypeAssistantFilter extends AbstractAssistantFilter
{
  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#getTitle()
   */
  public function getTitle() {
    return "Content type";
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#getDescription()
   */
  public function getDescription() {
    return "Content type filter";
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#isBoostAble()
   */
  public function isBoostAble() {
    return TRUE;
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#_build($context, $values, $query)
   */
  protected function _build(AbstractAssistantContext $context, &$values, SolrQuery $query) {
    if (! empty($values['types'])) {
      $fields = new SolrTermCollection();
      $fields->setOperator(SolrOperator::OPERATOR_OR);
      foreach ($values['types'] as $node_type => $enabled) {
        if ($enabled) {
          $fields->add($node_type);
        }
      }
      $fieldQuery = new SolrFieldFilter('type', $fields);
      $fieldQuery->setBoost($values['boost']);
      $fieldQuery->setExclusion(SolrOperator::OPERATOR_REQUIRE);
      $query->fq->add($fieldQuery);
    }
  }
  
  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#_form($context, $values)
   */
  protected function _form(AbstractAssistantContext $context, &$values = array()) {
    $form = array();
    $form['types'] = array(
      '#type' => 'checkboxes',
      '#options' => node_get_types('names'),
      '#default_value' => $values['types'] ? $values['types'] : array(),
      '#required' => TRUE,
    );
    return $form;
  }

  /**
   * (non-PHPdoc)
   * @see AbstractAssistantFilter#isFilterQuery()
   */
  public function isFilterQuery() {
    return TRUE;
  }
}
