<?php

namespace Drupal\association_menu\Form;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\association\Entity\AssociationInterface;
use Drupal\association\Entity\AssociationLink;
use Drupal\association\Entity\AssociationPage;
use Drupal\association_menu\AssociationMenuStorageInterface;

/**
 * Form for managing the menu built for association content.
 */
class AssociationMenuForm extends FormBase implements ContainerInjectionInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Association menu item storage manager.
   *
   * @var \Drupal\association_menu\AssociationMenuStorageInterface
   */
  protected $menuStorage;

  /**
   * Create a new instance of the entity association menu management form.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\association_menu\AssociationMenuStorageInterface $association_menu_storage
   *   Association navigation storage manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, AssociationMenuStorageInterface $association_menu_storage) {
    $this->entityTypeManager = $entity_type_manager;
    $this->menuStorage = $association_menu_storage;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.manager'),
      $container->get('association_menu.storage')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'association_menu_management_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, AssociationInterface $association = NULL) {
    if (!$association) {
      throw new NotFoundHttpException();
    }

    $form_state->set('association_id', $association->id());

    $form['#title'] = $this->t('Menu for @bundle_label: %label', [
      '@bundle_label' => $association->getType()->label(),
      '%label' => $association->label(),
    ]);

    $form['menu_tree'] = [
      '#type' => 'table',
      '#attributes' => ['id' => 'draggable-nav-items-overview'],
      '#empty' => $this->t('No content associated to this relationship.'),
      '#header' => [
        'label' => $this->t('Title'),
        'type' => $this->t('Type'),
        'enabled' => $this->t('Enabled'),
        'expanded' => $this->t('Expanded'),
        'parent' => $this->t('Parent'),
        'weight' => $this->t('Sort order'),
        'actions' => $this->t('Operations'),
      ],
      '#tabledrag' => [
        'parent' => [
          'action' => 'match',
          'relationship' => 'parent',
          'group' => 'association__nav-parent',
          'source' => 'association__nav-id',
          'hidden' => TRUE,
        ],
        'sort' => [
          'action' => 'order',
          'relationship' => 'sibling',
          'group' => 'association__nav-weight',
        ],
      ],
    ];

    $menu = $this->menuStorage->getMenuItems($association);
    foreach ($menu as $item) {
      $this->addRows($association, $item, $form['menu_tree']);
    }

    $form['actions'] = [
      '#type' => 'actions',

      'save' => [
        '#type' => 'submit',
        '#value' => $this->t('Save'),
      ],
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $items = $form_state->getValue('menu_tree');
    $assocId = $form_state->get('association_id');
    $association = $this->entityTypeManager
      ->getStorage('association')
      ->load($assocId);

    if ($association && $items) {
      foreach ($items as $id => &$data) {
        $data['id'] = $id;

        if (empty($data['parent'])) {
          $data['parent'] = NULL;
        }
        unset($data['label']);
      }
      unset($data);

      // Update just the menu tree related fields of the menu links.
      $this->menuStorage->updateMenuTree($association, $items);
      $this
        ->messenger()
        ->addStatus($this->t('Successfully updated menu configurations'));
    }
    else {
      $this
        ->messenger()
        ->addError($this->t('Unable to save menu changes.'));
    }
  }

  /**
   * Build the table row, and subtree rows for the navigation tree.
   *
   * @param \Drupal\association\Entity\AssociationInterface $association
   *   The association this menu item belongs to.
   * @param object $item
   *   Menu item data to build into the data row.
   * @param array $table
   *   Reference to the table form element.
   * @param int $depth
   *   The current depth to build the rows at (controls the indents).
   */
  protected function addRows(AssociationInterface $association, object $item, array &$table, $depth = 0) {
    $itemId = $item->id;
    $title = $item->title;
    $behavior = $association->getBehavior();

    // If not title set from the data, set a default from the entity if
    // one is available, or create a placeholder.
    if (empty($title)) {
      $title = (!empty($item->entity) && $item->entity instanceof EntityInterface)
        ? $item->entity->label()
        : $this->t('<missing title>');
    }
    $operations = [];
    $operations['edit'] = [
      'title' => $this->t('Edit'),
      'url' => Url::fromRoute('association_menu.edit_item_form', [
        'association' => $item->association,
        'menu_item_id' => $item->id,
      ]),
    ];

    if (empty($item->entity)) {
      $operations['delete'] = [
        'title' => $this->t('Delete'),
        'url' => Url::fromRoute('association_menu.delete_item_confirm', [
          'association' => $item->association,
          'menu_item_id' => $item->id,
        ]),
      ];

      $linkType = $this->t('Custom menu link');
    }
    elseif ($item->entity instanceof AssociationPage) {
      $linkType = $this->t('@bundle_label main page', [
        '@bundle_label' => $association->getType()->label(),
      ]);
    }
    elseif ($behavior && $item->entity instanceof AssociationLink) {
      $linkType = $behavior->getTagLabel($item->entity->getTag());
    }
    else {
      $linkType = $this->t('@bundle_label content', [
        '@bundle_label' => $association->getType()->label(),
      ]);
    }

    $table[$itemId] = [
      '#attributes' => [
        'id' => $itemId,
        'class' => ['draggable'],
      ],
      'label' => [
        'indent' => [
          '#theme' => 'indentation',
          '#size' => $depth,
        ],
        'display_name' => [
          '#type' => 'link',
          '#title' => $title,
          '#url' => $item->url,
        ],
        'id' => [
          '#type' => 'hidden',
          '#value' => $itemId,
          '#attributes' => [
            'class' => ['association__nav-id'],
          ],
        ],
      ],
      'type' => [
        '#markup' => $linkType,
      ],
      'enabled' => [
        '#type' => 'checkbox',
        '#default_value' => $item->enabled,
      ],
      'expanded' => [
        '#type' => 'checkbox',
        '#default_value' => $item->expanded,
      ],
      'parent' => [
        '#type' => 'textfield',
        '#default_value' => $item->parent,
        '#attributes' => [
          'class' => ['association__nav-parent'],
        ],
      ],
      'weight' => [
        '#type' => 'number',
        '#default_value' => $item->nav_weight->value ?? 0,
        '#attributes' => [
          'class' => ['association__nav-weight'],
        ],
      ],
      'actions' => [
        '#type' => 'operations',
        '#links' => $operations,
      ],
    ];

    if (!empty($item->children)) {
      foreach ($item->children as $child) {
        $this->addRows($association, $child, $table, $depth + 1);
      }
    }
  }

}
