<?php

namespace Drupal\association_menu\Plugin\Block;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Plugin\Context\ContextInterface;
use Drupal\Core\Block\BlockBase;
use Drupal\association_menu\AssociationMenuStorageInterface;
use Drupal\association_menu\AssociationMenuBuilderInterface;

/**
 * Creates a block plugin, which displays association navigation.
 *
 * @Block(
 *   id = "association_menu",
 *   admin_label = @Translation("Association menu"),
 *   category = @Translation("Entity association"),
 *   context_definitions = {
 *     "association" = @ContextDefinition("entity:association", label = @Translation("Entity association")),
 *   },
 * )
 */
class AssociationMenuBlock extends BlockBase implements ContainerFactoryPluginInterface {

  /**
   * Fetches and saves manager for association navigation menu items.
   *
   * @var \Drupal\association_menu\AssociationMenuStorageInterface
   */
  protected $menuStorage;

  /**
   * Builds association navigation data into renderable menu.
   *
   * @var \Drupal\association_menu\AssociationMenuBuilderInterface
   */
  protected $menuBuilder;

  /**
   * Create a new instance of a AssociationMenuBlock block plugin.
   *
   * @param array $configuration
   *   The block configuration.
   * @param string $plugin_id
   *   The block plugin identifier.
   * @param mixed $plugin_definition
   *   The block plugin definition.
   * @param \Drupal\association_menu\AssociationMenuStorageInterface $association_menu_storage
   *   Fetches and saves manager for association menu items.
   * @param \Drupal\association_menu\AssociationMenuBuilderInterface $association_menu_builder
   *   Builds association menu data into renderable array.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, AssociationMenuStorageInterface $association_menu_storage, AssociationMenuBuilderInterface $association_menu_builder) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->menuStorage = $association_menu_storage;
    $this->menuBuilder = $association_menu_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('association_menu.storage'),
      $container->get('association_menu.builder')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    $cacheTags = parent::getCacheTags();
    $context = $this->getContext('association');

    if ($context instanceof ContextInterface) {
      $cacheTags = Cache::mergeTags($cacheTags, $context->getCacheTags());

      if ($association = $context->getContextValue()) {
        $menu = $this->menuStorage->getMenu($association);

        if (!empty($menu['cache'])) {
          $cacheTags = Cache::mergeTags($cacheTags, $menu['cache']->getCacheTags());
        }
      }
    }

    return $cacheTags;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheContexts() {
    $cacheContexts = parent::getCacheContexts();
    $context = $this->getContext('association');

    if ($context instanceof ContextInterface) {
      $cacheContexts = Cache::mergeContexts($cacheContexts, $context->getCacheContexts());

      if ($association = $context->getContextValue()) {
        $menu = $this->menuStorage->getMenu($association);

        if (!empty($menu['cache'])) {
          $cacheContexts = Cache::mergeTags($cacheContexts, $menu['cache']->getCacheContexts());
        }
      }
    }

    // Cache by route because each route can change which menu items end up in
    // the active trail.
    $cacheContexts = Cache::mergeContexts($cacheContexts, ['route']);
    return $cacheContexts;
  }

  /**
   * {@inheritdoc}
   */
  public function blockAccess(AccountInterface $account) {
    $association = $this->getContextValue('association');

    if ($association && $association->isActive()) {
      return AccessResult::allowed()
        ->addCacheableDependency($association);
    }

    return AccessResult::forbidden();
  }

  /**
   * {@inheritdoc}
   */
  public function build() {
    $association = $this->getContextValue('association');
    $menu = $this->menuStorage->getMenu($association);

    return [
      'menu' => $this->menuBuilder->buildMenu($association, $menu),
    ];
  }

}
