<?php

namespace Drupal\association\Entity\Form;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Plugin\PluginWithFormsInterface;
use Drupal\Core\Plugin\PluginFormFactoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Form\SubformState;
use Drupal\association\Plugin\BehaviorInterface;
use Drupal\association\Plugin\BehaviorPluginManagerInterface;

/**
 * Form for editing and creating new association types.
 */
class AssociationTypeForm extends EntityForm {

  /**
   * Service providing entity bundle information.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $entityBundleInfo;

  /**
   * The form factory for getting form objects for plugins operations.
   *
   * @var \Drupal\Core\Plugin\PluginFormFactoryInterface
   */
  protected $pluginFormFactory;

  /**
   * Association type behavior plugin manager.
   *
   * @var \Drupal\association\Plugin\BehaviorPluginManagerInterface
   */
  protected $behaviorManager;

  /**
   * Create a new instance of the AssociationTypeForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_bundle_info
   *   Service providing entity bundle information.
   * @param \Drupal\Core\Plugin\PluginFormFactoryInterface $plugin_form_factory
   *   The form factory for getting form objects for plugins operations.
   * @param \Drupal\association\Plugin\BehaviorPluginManagerInterface $behavior_manager
   *   Association type behavior plugin manager.
   */
  public function __construct(EntityTypeBundleInfoInterface $entity_bundle_info, PluginFormFactoryInterface $plugin_form_factory, BehaviorPluginManagerInterface $behavior_manager) {
    $this->entityBundleInfo = $entity_bundle_info;
    $this->pluginFormFactory = $plugin_form_factory;
    $this->behaviorManager = $behavior_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('entity_type.bundle.info'),
      $container->get('plugin_form.factory'),
      $container->get('plugin.manager.association.behavior')
    );
  }

  /**
   * Check to see if an ID is already in use for association type bundle.
   *
   * @return bool
   *   The boolean to indicate if this ID is already in use. TRUE if it already
   *   exists, and FALSE otherwise.
   */
  public function exists($id) {
    if ($id === 'add') {
      return TRUE;
    }

    $entityType = $this->getEntity()->getEntityType();
    $entityIds = $this->entityTypeManager->getStorage($entityType->id())
      ->getQuery()
      ->condition($entityType->getKey('id'), $id)
      ->execute();

    return (bool) $entityIds;
  }

  /**
   * Get a list of behavior plugins available for use with association types.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup|string[]
   *   An array of behavior plugin labels, keyed by the plugin ID.
   */
  protected function getPluginOptions() {
    $options = [];

    foreach ($this->behaviorManager->getDefinitions() as $id => $definition) {
      $options[$id] = $definition['label'];
    }

    return $options;
  }

  /**
   * Get an instance of the plugin configuration form if there is one.
   *
   * @param \Drupal\association\Plugin\BehaviorInterface $plugin
   *   The association behavior plugin to generate a configuration form for.
   *
   * @return \Drupal\Core\Plugin\PluginFormInterface|null
   *   A loaded plugin object if the plugin has a configuration form.
   */
  protected function getConfigurePluginForm(BehaviorInterface $plugin) {
    if ($plugin instanceof PluginFormInterface) {
      return $plugin;
    }

    if ($plugin instanceof PluginWithFormsInterface) {
      try {
        return $this->pluginFormFactory->createInstance($plugin, 'configure');
      }
      catch (InvalidPluginDefinitionException $e) {
        // Plugin does not have this form operation defined.
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);
    $entity = $this->entity;
    $hasData = $entity->hasData();

    $form += ['#parents' => []];

    if ($hasData) {
      $form['warning'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('This association type has data!'),
        '#attributes' => [
          'class' => ['message', 'message--warning'],
        ],
        '#markup' => $this->t('This Entity Association type has data (association instances) and will therefore lock configuration options that should not be changed because they can break existing content. You must remove all @label associations to edit these settings.', [
          '@label' => $entity->label(),
        ]),
      ];
    }

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $entity->label(),
      '#description' => $this->t("Label for the Entity Association type."),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity->id(),
      '#machine_name' => [
        'source' => ['label'],
        'exists' => [$this, 'exists'],
      ],
      '#disabled' => !$entity->isNew(),
    ];

    $form['hasPage'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('This Entity Association type has a dedicated landing page'),
      '#disabled' => $hasData,
      '#default_value' => $entity->get('hasPage'),
    ];

    $form['newRevision'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Create new page revisions by default'),
      '#default_value' => $entity->get('newRevision'),
      '#states' => [
        'invisible' => [
          ':input[name="hasPage"]' => ['unchecked' => TRUE],
        ],
      ],
    ];

    $form['searchable'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Is associated content searchable?'),
      '#default_value' => $entity->isContentSearchable(),
    ];

    $form['behavior'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Association behavior'),
      '#tree' => TRUE,

      'id' => [
        '#type' => 'select',
        '#title' => $this->t('Behavior plugin'),
        '#required' => TRUE,
        '#access' => !$hasData,
        '#options' => $this->getPluginOptions(),
        '#ajax' => [
          'wrapper' => 'association-type-behavior-configuration',
          'callback' => static::class . '::pluginConfigurationAjax',
        ],
      ],
      'config' => [
        '#parents' => ['plugin', 'config'],
        '#array_parents' => ['plugin', 'config'],
      ],
    ];

    if ($plugin = $entity->getBehavior()) {
      $form['behavior']['id']['#default_value'] = $plugin->getPluginId();

      if ($pluginFormInstance = $this->getConfigurePluginForm($plugin)) {
        $pluginForm = &$form['behavior']['config'];
        $subformState = SubformState::createForSubform($pluginForm, $form, $form_state);
        $pluginForm = $pluginFormInstance->buildConfigurationForm($pluginForm, $subformState, $entity);

        // Apply the default form wrapping components if nothing was defined
        // internally by the plugin configuration form itself.
        $pluginForm += [
          '#type' => 'fieldset',
          '#title' => $this->t('Configurations'),
        ];
      }
      else {
        $form['behavior']['config']['#value'] = [];
      }
    }

    // Add a consistent wrapper around the configuration form elements.
    $form['behavior']['config']['#prefix'] = '<div id="association-type-behavior-configuration">';
    $form['behavior']['config']['#suffix'] = '</div>';

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    $pluginId = $form_state->getValue(['behavior', 'id']);

    if (!empty($pluginId)) {
      try {
        $config = [];
        $entity = $this->getEntity();
        $current = $this->entityTypeManager
          ->getStorage('association_type')
          ->load($entity->id());

        if ($current) {
          $currentPlugin = $current->get('behavior');
          $config = $currentPlugin['id'] === $pluginId ? $currentPlugin['config'] : [];
        }

        $behavior = $this->behaviorManager->createInstance($pluginId, $config);

        if ($pluginFormInstance = $this->getConfigurePluginForm($behavior)) {
          $subformState = SubformState::createForSubform($form['behavior']['config'], $form, $form_state);
          $pluginFormInstance->validateConfigurationForm($form['behavior']['config'], $subformState);
        }
      }
      catch (\Exception $e) {
        $form_state->setError($form['behavior'], $this->t('Unable to use selected behavior plugin due to an internal error.'));

        drupal_watchdog('association', $e);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $entity = $this->entity;

    try {
      $plugin = $entity->getBehavior();

      // If this plugin uses a configuration form, allow the configuration
      // submit handler to properly handle building the configuration values.
      if ($pluginFormInstance = $this->getConfigurePluginForm($plugin)) {
        $subformState = SubformState::createForSubform($form['behavior']['config'], $form, $form_state);
        $pluginFormInstance->submitConfigurationForm($form['behavior']['config'], $subformState);

        $entity->set('behavior', [
          'id' => $plugin->getPluginId(),
          'config' => $plugin->getConfiguration(),
        ]);
      }

      // Save the bundle entity changes.
      $status = $entity->save();

      $msgParams = ['%label' => $entity->label()];
      $msg = SAVED_NEW === $status
        ? $this->t('Created the %label Entity Association type.', $msgParams)
        : $this->t('Saved the %label Entity Association type changes.', $msgParams);

      $this->messenger()->addStatus($msg);

      $form_state->setRedirectUrl($entity->toUrl('collection'));
    }
    catch (\Exception $e) {
      $this->messenger()->addError($this->t('Unable to have changes to the Entity Association type settings.'));
    }

    return $status;
  }

  /**
   * AJAX callback for when the association type behavior is changed.
   *
   * @param array $form
   *   Complete form structure and element definitions.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Current state, build information and user input for the form.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse|array
   *   A set of AJAX commands or a renderable array with the form changes to
   *   perform through AJAX.
   */
  public static function pluginConfigurationAjax(array $form, FormStateInterface $form_state) {
    return $form['behavior']['config'];
  }

}
