<?php

namespace Drupal\association\Entity;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\ContentEntityBase;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityPublishedTrait;
use Drupal\Core\Url;
use Drupal\user\UserInterface;
use Drupal\association\Plugin\BehaviorInterface;

/**
 * Entity which maintains content entity associations.
 *
 * This is the entity from which linked content and association page content
 * is associated to. The removal of this entity will trigger the deletion of
 * those dependent resources (pages and link entities).
 *
 * @ContentEntityType(
 *   id = "association",
 *   label = @Translation("Entity association"),
 *   label_plural = @Translation("Entity associations"),
 *   bundle_label = @Translation("Association type"),
 *   bundle_entity_type = "association_type",
 *   base_table = "association",
 *   data_table = "association_field_data",
 *   token_type = "association",
 *   field_ui_base_route = "entity.association_type.edit_form",
 *   admin_permission = "administer association configurations",
 *   permission_granularity = "bundle",
 *   translatable = TRUE,
 *   fieldable = TRUE,
 *   entity_keys = {
 *     "id" = "id",
 *     "bundle" = "type",
 *     "label" = "name",
 *     "uid" = "uid",
 *     "status" = "status",
 *     "published" = "status",
 *     "langcode" = "langcode",
 *   },
 *   handlers = {
 *     "access" = "Drupal\association\Entity\Access\AssociationAccessControlHandler",
 *     "list_builder" = "Drupal\association\Entity\Controller\AssociationListBuilder",
 *     "views_data" = "Drupal\association\Entity\Views\AssociationViewsData",
 *     "translation" = "Drupal\content_translation\ContentTranslationHandler",
 *     "storage" = "Drupal\association\Entity\Storage\AssociationStorage",
 *     "form" = {
 *       "default" = "Drupal\association\Entity\Form\AssociationForm",
 *       "edit" = "Drupal\association\Entity\Form\AssociationForm",
 *       "delete" = "Drupal\association\Entity\Form\AssociationDeleteConfirm",
 *       "delete-multiple-confirm" = "Drupal\association\Entity\Form\AssociationDeleteMultipleConfirm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\association\Entity\Routing\AssociationHtmlRouteProvider",
 *     },
 *   },
 *   links = {
 *     "collection" = "/admin/content/association",
 *     "add-page" = "/association/add",
 *     "add-form" = "/association/add/{association_type}",
 *     "canonical" = "/association/{association}/manage",
 *     "manage" = "/association/{association}/manage",
 *     "edit-form" = "/association/{association}/edit",
 *     "delete-form" = "/association/{association}/delete",
 *     "delete-multiple-form" = "/admin/content/association/delete-multiple",
 *   },
 * )
 */
class Association extends ContentEntityBase implements AssociationInterface, EntityPublishedInterface {

  use EntityChangedTrait;
  use EntityPublishedTrait;

  /**
   * The association page belonging to this entity, if landing page is enabled.
   *
   * This field is boolean FALSE if there is no companion page available for
   * this entity. There should only be a single landing page per association.
   *
   * @var \Drupal\association\Entity\AssociationPage|bool
   */
  protected $associationPage;

  /**
   * The purging state of this entity association. True during deletion.
   *
   * @var bool
   */
  protected $isPurging = FALSE;

  /**
   * Default value callback for 'uid' base field definition.
   *
   * @return array
   *   An array containing the default author data.
   */
  public static function getCurrentUserId() {
    return [
      \Drupal::currentUser()->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function isPurging() {
    return $this->isPurging;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {
    if ($this->cacheTags) {
      return Cache::mergeTags(parent::getCacheTagsToInvalidate(), $this->cacheTags);
    }

    return parent::getCacheTagsToInvalidate();
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTagsToInvalidate() {
    if ($this->isNew()) {
      return [];
    }

    // Invalidate the associated content list for the parent association.
    $tags[] = 'association:links:' . $this->id();
    return Cache::mergeTags($tags, parent::getCacheTagsToInvalidate());
  }

  /**
   * {@inheritdoc}
   */
  public function getType() {
    return $this->type->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getBehavior(): ?BehaviorInterface {
    return $this->getType()->getBehavior();
  }

  /**
   * {@inheritdoc}
   */
  public function getOwner() {
    return $this->get('uid')->entity;
  }

  /**
   * {@inheritdoc}
   */
  public function getOwnerId() {
    return $this->get('uid')->target_id;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwnerId($uid) {
    $this->set('uid', $uid);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function setOwner(UserInterface $account) {
    $this->set('uid', $account->id());
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getCreatedTime() {
    return $this->get('created')->value;
  }

  /**
   * {@inheritdoc}
   */
  public function setCreatedTime($timestamp) {
    $this->set('created', $timestamp);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isActive() {
    return $this->status->value;
  }

  /**
   * {@inheritdoc}
   */
  public function getPage() {
    if (!isset($this->associationPage)) {
      $type = $this->getType();

      if ($type && $type->hasPage()) {
        $page = \Drupal::entityTypeManager()
          ->getStorage('association_page')
          ->load($this->id());
      }

      $this->associationPage = $page ?? FALSE;
    }

    return $this->associationPage;
  }

  /**
   * {@inheritdoc}
   */
  public function toUrl($rel = 'canonical', array $options = []) {
    if ($rel === 'canonical') {
      // If there is a companion page to this association, use the page for the
      // canonical URL instead of linking to the association directly.
      if ($page = $this->getPage()) {
        return $page->toUrl('canonical', $options);
      }
      elseif ($this->getType()->hasPage() && !$this->isNew()) {
        return Url::fromRoute('entity.association_page.canonical', [
          'association_page' => $this->id(),
        ]);
      }

      // If there is no page, then the canonical URL is actually the content
      // management URL.
      $rel = 'manage';
    }

    return parent::toUrl($rel, $options);
  }

  /**
   * {@inheritdoc}
   */
  public function postSave(EntityStorageInterface $storage, $update = FALSE) {
    parent::postSave($storage, $update);

    // Ensure that the calculated field is loaded before Pathauto has a
    // chance to update this data in the database.
    if ($this->original && $this->original->hasField('path')) {
      $this->original->path->alias;
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function preDelete(EntityStorageInterface $storage, array $entities) {
    parent::preDelete($storage, $entities);

    $ids = [];
    foreach ($entities as $entity) {
      $entity->isPurging = TRUE;
      $ids[] = $entity->id();
    }

    // @todo Convert this into a worker queue and perform the delete of
    // associations and possibily the related content off-line and in a batch.
    if ($ids) {
      // Delete companion pages if there are any.
      $pageStorage = \Drupal::entityTypeManager()
        ->getStorage('association_page');

      if ($pages = $pageStorage->loadMultiple($ids)) {
        $pageStorage->delete($pages);
      }

      $linkStorage = \Drupal::entityTypeManager()
        ->getStorage('association_link');

      $assocIds = $linkStorage->getQuery()
        ->condition('association', $ids, 'IN')
        ->execute();

      if ($links = $linkStorage->loadMultiple($assocIds)) {
        $linkStorage->delete($links);
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {
    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['uid'] = BaseFieldDefinition::create('entity_reference')
      ->setTargetEntityTypeId($entity_type->id())
      ->setLabel(t('Authored by'))
      ->setDescription(t('The author and owner of this content.'))
      ->setRevisionable(TRUE)
      ->setTranslatable(FALSE)
      ->setSetting('target_type', 'user')
      ->setSetting('handler', 'default')
      ->setDefaultValueCallback(static::class . '::getCurrentUserId')
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 5,
        'settings' => [
          'match_operator' => 'CONTAINS',
          'size' => '60',
          'autocomplete_type' => 'tags',
          'placeholder' => '',
        ],
      ]);

    $fields['name'] = BaseFieldDefinition::create('string')
      ->setTargetEntityTypeId($entity_type->id())
      ->setLabel(t('Name'))
      ->setDescription(t('The name of this content association.'))
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setRequired(TRUE)
      ->setDefaultValue('')
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setSettings([
        'max_length' => 255,
        'text_processing' => 0,
      ])
      ->setDisplayOptions('view', [
        'label' => 'hidden',
        'type' => 'string',
        'weight' => -4,
      ])
      ->setDisplayOptions('form', [
        'type' => 'string_textfield',
        'weight' => -4,
      ]);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setTargetEntityTypeId($entity_type->id())
      ->setLabel(t('Active'))
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setDefaultValue(TRUE)
      ->setDisplayConfigurable('view', FALSE)
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'weight' => -3,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setTargetEntityTypeId($entity_type->id())
      ->setLabel(t('Created'))
      ->setRevisionable(FALSE)
      ->setTranslatable(FALSE)
      ->setDescription(t('The time of created.'));

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setTargetEntityTypeId($entity_type->id())
      ->setLabel(t('Changed'))
      ->setRevisionable(TRUE)
      ->setTranslatable(TRUE)
      ->setDescription(t('The last updated date.'));

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public static function bundleFieldDefinitions(EntityTypeInterface $entity_type, $bundle, array $base_field_definitions) {
    $fields = [];

    try {
      $bundle = \Drupal::entityTypeManager()
        ->getStorage($entity_type->getBundleEntityType())
        ->load($bundle);

      if ($bundle) {
        $fields['name'] = clone $base_field_definitions['name'];
        $fields['name']->setDescription(t('The name of this @bundle_name.', [
          '@bundle_name' => $bundle->label(),
        ]));

        $fields['status'] = clone $base_field_definitions['status'];
        $fields['status']->setDescription(t('Is this @bundle_name enabled and publishing its content.', [
          '@bundle_name' => $bundle->label(),
        ]));
      }
    }
    catch (InvalidPluginDefinitionException $e) {
      // Unlikely that this would happen, but this would happen when the
      // association_type entity definition is missing or the storage handler.
    }

    return $fields;
  }

}
