<?php

namespace Drupal\association\Access;

use Symfony\Component\Routing\Route;
use Drupal\Core\Routing\Access\AccessInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\node\NodeInterface;
use Drupal\node\Access\NodeRevisionAccessCheck;
use Drupal\association\AssociationNegotiatorInterface;
use Drupal\association\EntityAdapterManagerInterface;
use Drupal\association\Entity\AssociationInterface;

/**
 * Decorates the node revision access check and adds association access rules.
 *
 * Decorator combines the original node revision permission with the entity
 * association "manage" content permission to allow or deny revision management.
 */
class AssociatedNodeRevisionAccess extends NodeRevisionAccessCheck implements AccessInterface {

  /**
   * The node revision access checker being decorated.
   *
   * @var \Drupal\Core\Routing\AccessInterface
   */
  protected $accessCheck;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The negotiator to determine the owning association by the node context.
   *
   * @var \Drupal\association\AssociationNegotiatorInterface
   */
  protected $negotiator;

  /**
   * The association entity adapter manager.
   *
   * @var \Drupal\association\EntityAdapterManagerInterface
   */
  protected $adapterManager;

  /**
   * Creates a new instance of the AssociatedNodeRevisionAccess decorator class.
   *
   * @param \Drupal\Core\Routing\Access\AccessInterface $access_check
   *   The node revision access checker being decorated.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\association\AssociationNegotiatorInterface $association_negotiator
   *   The negotiator to determine the owning association by the node context.
   * @param \Drupal\association\EntityAdapterManagerInterface $entity_adapter_manager
   *   The association entity adapter manager.
   */
  public function __construct(AccessInterface $access_check, EntityTypeManagerInterface $entity_type_manager, AssociationNegotiatorInterface $association_negotiator, EntityAdapterManagerInterface $entity_adapter_manager) {
    $this->accessCheck = $access_check;
    $this->adapterManager = $entity_adapter_manager;
    $this->negotiator = $association_negotiator;
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * Finds teh association which this entity belongs if it has one.
   *
   * @param \Drupal\Core\Entity\EntityInterface $entity
   *   The entity to find the association entity for.
   *
   * @return \Drupal\association\Entity\AssociationInterface|null
   *   Get the association entity which owns this associated entity, or NULL
   *   if the node is not part of an entity association.
   */
  protected function getAssociation(EntityInterface $entity): ?AssociationInterface {
    if ($this->adapterManager->isAssociable($entity)) {
      // Because this is potentially a revision, avoid loading the associations
      // field with values, and instead get the values directly from storage.
      $links = $this->entityTypeManager
        ->getStorage('association_link')
        ->loadByEntityInfo($entity->getEntityTypeId(), $entity->id());

      return !empty($links) ? reset($links)->getAssociation() : NULL;
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function access(Route $route, AccountInterface $account, $node_revision = NULL, NodeInterface $node = NULL) {
    // Perform the original node revision access check.
    $access = $this->accessCheck->access($route, $account, $node_revision, $node);

    // If there was an association combine the results, mostly to ensure we
    // have proper caching from both access results.
    if ($node && ($association = $this->getAssociation($node))) {
      $associationAccess = $association->access('manage', $account, TRUE);
      $access = $access->andIf($associationAccess);
    }

    return $access;
  }

  /**
   * {@inheritdoc}
   */
  public function checkAccess(NodeInterface $node, AccountInterface $account, $op = 'view') {
    // Perform the original node revision access check.
    $access = $this->checkAccess($node, $account, $op);

    // If there was an association combine the results, mostly to ensure we
    // have proper caching from both access results.
    if ($association = $this->negotiator->byEntity($node)) {
      $associationAccess = $association->access('manage', $account, TRUE);
      $access = $access->andIf($associationAccess);
    }

    return $access;
  }

}
