<?php

namespace Drupal\association\Plugin\Association\Behavior;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginFormInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\association\EntityAdapterManagerInterface;
use Drupal\association\Plugin\BehaviorInterface;
use Drupal\association\Entity\AssociationInterface;
use Drupal\association\Entity\AssociationTypeInterface;

/**
 * Behavior for managing associations that have a entity list.
 *
 * Manages entities linked to association and dictates the controls, which
 * entities/bundles are allowed. The behavior also is responsible for building
 * the admin UI for the association management.
 *
 * @AssociationBehavior(
 *   id = "entity_list",
 *   label = @Translation("Entity list"),
 *   manager_builder = "Drupal\association\Behavior\Manager\EntityListBuilder",
 * )
 */
class EntityListBehavior extends PluginBase implements BehaviorInterface, PluginFormInterface, ContainerFactoryPluginInterface {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity adapter manager for fetching entity type specific handler.
   *
   * @var \Drupal\association\EntityAdapterManagerInterface
   */
  protected $adapterManager;

  /**
   * List of labels for behavior tags, keyed by the tag identifier.
   *
   * @var \Drupal\Core\StringTranslation\TranslatableMarkup[]|string[]
   */
  protected $tagLabels = [];

  /**
   * Create a new instance of the ContentManifestBehavior plugin.
   *
   * @param array $configuration
   *   Plugin configuration options.
   * @param string $plugin_id
   *   The plugin identifier.
   * @param mixed $plugin_definition
   *   The plugin definition (from plugin discovery).
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\association\EntityAdapterManagerInterface $entity_adapter_manager
   *   The entity adapter manager for fetching entity type specific handler.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, EntityAdapterManagerInterface $entity_adapter_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->entityTypeManager = $entity_type_manager;
    $this->adapterManager = $entity_adapter_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.association.entity_adapter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'entity_types' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration() {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration) {
    $this->configuration = $configuration + $this->defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityTypes() {
    $entityTypes = $this->getConfiguration()['entity_types'] ?? [];
    $adapterTypes = $this->adapterManager->getEntityTypes();

    return array_intersect_key($adapterTypes, $entityTypes);
  }

  /**
   * {@inheritdoc}
   */
  public function getTagLabel($tag) {
    if (!isset($this->tagLabels[$tag])) {
      $this->tagLabels[$tag] = FALSE;
      list($entityTypeId, $bundle) = explode('-', $tag, 2);

      if ($adapter = $this->adapterManager->getAdapterByEntityType($entityTypeId)) {
        $bundles = $adapter->getBundles();

        if (!empty($bundles[$bundle])) {
          $this->tagLabels[$tag] = new FormattableMarkup('@entity_type_label: @bundle_label', [
            '@entity_type_label' => $adapter->getLabel(),
            '@bundle_label' => $bundles[$bundle],
          ]);
        }
      }
    }

    return $this->tagLabels[$tag] ?: '';
  }

  /**
   * {@inheritdoc}
   */
  public function getManagerBuilderClass() {
    return $this->pluginDefinition['manager_builder'];
  }

  /**
   * {@inheritdoc}
   */
  public function createAccess(AssociationInterface $association, $tag, AccountInterface $account) {
    list($entityTypeId, $bundle) = explode('-', $tag, 2);
    $entityTypes = $this->getConfiguration()['entity_types'] ?? [];

    $accessResult = !empty($entityTypes[$entityTypeId][$bundle])
      ? AccessResult::allowed()
      : AccessResult::forbidden();

    return $accessResult->addCacheableDependency($association->getType());
  }

  /**
   * {@inheritdoc}
   */
  public function createTagEntity(AssociationInterface $association, $tag) {
    list($entityTypeId, $bundle) = explode('-', $tag, 2);
    $entityTypes = $this->getConfiguration()['entity_types'] ?? [];

    if (!empty($entityTypes[$entityTypeId][$bundle])) {
      try {
        return $this->adapterManager
          ->getAdapterByEntityType($entityTypeId)
          ->createEntity($bundle);
      }
      catch (PluginException $e) {
        // Unable to fetch the entity adapter for this entity type and
        // therefore we should fail to create the entity.
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigUpdate(AssociationTypeInterface $assocation_type, array $changes) {
    $errors = [];
    $current = $this->getConfiguration()['entity_types'] ?? [];
    $entityTypes = $changes['entity_types'];

    if ($diff = array_diff($current, $entityTypes)) {
      $errors[] = $this->t('Cannot remove entity types from entity list behavior when there is data: @list', [
        '@list' => implode(', ', $diff),
      ]);
    }
    else {
      foreach ($current as $type => $bundles) {
        if ($bundleDiff = array_diff($bundles, $entityTypes[$type])) {
          $errors[] = $this->t('Cannot remove @type bundles (@list) when Entity Association has content.', [
            '@type' => $type,
            '@list' => implode(', ', $bundleDiff),
          ]);
        }
      }
    }

    return [];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state, AssociationTypeInterface $association_type = NULL) {
    $entityTypes = $this->getConfiguration()['entity_types'] ?? [];
    $form_state->set('association_type', $association_type->id());

    foreach ($this->adapterManager->getEntityTypes() as $type) {
      try {
        $adapter = $this->adapterManager->getAdapterByEntityType($type);

        $form['entity_types'][$type] = [
          '#type' => 'checkboxes',
          '#title' => $this->t('Allowed @label types', [
            '@label' => $adapter->getLabel(),
          ]),
          '#options' => $adapter->getBundles(),
          '#default_value' => $entityTypes[$type] ?? [],
        ];
      }
      catch (PluginException $e) {
        // Unable to create the association entity adapter for the entity type.
        // Skip as this probably means a providing module is missing.
      }
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigurationForm(array &$form, FormStateInterface $form_state) {
    $current = $this->getConfiguration()['entity_types'] ?? [];
    $entityTypes = $form_state->getValue('entity_types');

    if ($typeId = $form_state->get('association_type')) {
      $type = $this->entityTypeManager->getStorage('association_type')->load($typeId);
      if (!$type->hasData()) {
        return;
      }
    }

    foreach ($current as $type => $bundles) {
      $updated = array_filter($entityTypes[$type]);

      // Check for missing entity bundles.
      if (empty($updated)) {
        $form_state->setError($form['entity_types'][$type], $this->t('Cannot remove @type bundles (@list) when entity association has content.', [
          '@type' => $type,
          '@list' => implode(', ', $bundles),
        ]));
      }
      elseif ($bundleDiff = array_diff($bundles, $updated)) {
        $form_state->setError($form['entity_types'][$type], $this->t('Cannot remove @type bundles (@list) when entity association has data.', [
          '@type' => $type,
          '@list' => implode(', ', $bundleDiff),
        ]));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state) {
    $entityTypes = $form_state->getValue('entity_types') ?? [];

    foreach ($entityTypes as &$bundles) {
      $bundles = array_filter($bundles);
    }

    $config = ['entity_types' => array_filter($entityTypes)];
    $this->setConfiguration($config);
  }

}
