<?php

namespace Drupal\association_menu;

use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Core\Access\AccessManagerInterface;
use Drupal\Core\Cache\Cache;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Database\DatabaseException;
use Drupal\Core\Database\Connection;
use Drupal\Core\Entity\EntityMalformedException;
use Drupal\Core\Entity\Exception\UndefinedLinkTemplateException;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Logger\LoggerChannelTrait;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Utility\Error;
use Drupal\Core\Url;
use Drupal\association\Entity\AssociationInterface;
use Drupal\association\Entity\AssociatedEntityInterface;

/**
 * The association storage for managing the menu link items.
 */
class AssociationMenuStorage implements AssociationMenuStorageInterface {

  use LoggerChannelTrait;

  /**
   * An array of cached navigation data, already sorted and access checked.
   *
   * @var array
   */
  protected $menus = [];

  /**
   * Database connection where the association navigation data is stored.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected $db;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The access manager.
   *
   * @var \Drupal\Core\Access\AccessManagerInterface
   */
  protected $accessManager;

  /**
   * The account to use for access checks by default.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $account;

  /**
   * Create a new instance of the AssociationMenuStorage manager class.
   *
   * @param \Drupal\Core\Database\Connection $database
   *   The database connection to use for storage and retrieval of menu items.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Access\AccessManagerInterface $access_manager
   *   The access manager.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account to use for access checks by default.
   */
  public function __construct(Connection $database, EntityTypeManagerInterface $entity_type_manager, AccessManagerInterface $access_manager, AccountInterface $account) {
    $this->db = $database;
    $this->entityTypeManager = $entity_type_manager;
    $this->accessManager = $access_manager;
    $this->account = $account;
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags(AssociationInterface $association): array {
    return [
      'association:menu:' . $association->id(),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function clearCache($associationId = NULL): void {
    if (isset($associationId)) {
      unset($this->menus[$associationId]);
    }
    else {
      $this->menus = [];
    }
  }

  /**
   * Get the loaded entity for a menu item, if it has entity property reference.
   *
   * @param object $item
   *   The menu item with "entity" property to load the entity from.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   The loaded entity if it could be loaded, or NULL if unable to load.
   */
  protected function getItemEntity(object $item): ?EntityInterface {
    if (empty($item->entity)) {
      return NULL;
    }

    [$entityType, $entityId] = explode(':', $item->entity);
    if ($entityType && $entityId) {
      try {
        return $this->entityTypeManager
          ->getStorage($entityType)
          ->load($entityId);
      }
      catch (InvalidPluginDefinitionException $e) {
        $this
          ->getLogger('association_menu')
          ->error('%type: @message in %function (line %line of %file).', Error::decodeException($e));
      }
    }
  }

  /**
   * Constructs the URL for a menu item.
   *
   * Menu items can contain the data for generating a URL in one of three ways
   * in order of preference:
   *  - An associated entity
   *  - Route name and parameters
   *  - URI for non-routed and external links
   * If unable to generate a proper URL then method returns NULL and the link
   * in general should be considered invalid. "<nolink>" and "<none>" can be
   * use as the route for, valid URLs that don't link to anything.
   *
   * @param object $item
   *   The menu item data.
   *
   * @return \Drupal\Core\Url|null
   *   The URL generated from the data in the menu item. NULL is returned
   *   if the method failed to determine a proper URL from the data.
   */
  protected function getItemUrl(object $item): ?Url {
    $url = NULL;

    // Build the URL from the menu link data preferring in order:
    // associated entity, route data, or URI string information to build from.
    if (!empty($item->entity)) {
      // If $item->entity is still a string, then it never loaded correctly.
      // This should be treated as NO URL is available.
      if ($item->entity instanceof EntityInterface) {
        try {
          $url = $item->entity->toUrl('canonical', $item->options);
        }
        catch (EntityMalformedException | UndefinedLinkTemplateException $e) {
          $this
            ->getLogger('association_menu')
            ->error('%type: @message in %function (line %line of %file).', Error::decodeException($e));
        }
      }
    }
    elseif (!empty($item->route)) {
      $route = unserialize($item->route, ['allowed_classes' => FALSE]);

      // Creates the URL based on internal Drupal routes and parameters.
      if (is_array($route) && !empty($route['route_name'])) {
        $route += ['route_parameters' => []];
        $url = Url::fromRoute($route['route_name'], $route['route_parameters'], $item->options);
      }
    }
    elseif (!empty($item->uri)) {
      // Non-route, or external links.
      $url = Url::fromUri($item->uri, $item->options);
    }

    return $url;
  }

  /**
   * Ensure the data is properly unserialized, and entity and URL is resolved.
   *
   * @param object $item
   *   The raw menu item data (usually from DB), to load and clean up. This
   *   object is updated and is also returned.
   *
   * @return object
   *   Returns the object parameter for chaining.
   */
  protected function loadItem(object $item): object {
    if ($entity = $this->getItemEntity($item)) {
      $item->entity = $entity;
    }

    $allowedTitleClasses = [
      '\Drupal\Component\Render\FormattableMarkup',
      '\Drupal\Core\StringTranslation\TranslatableMarkup',
    ];
    $item->title = unserialize($item->title, ['allowed_classes' => $allowedTitleClasses]) ?? '';
    $item->options = unserialize($item->options, ['allowed_classes' => FALSE]) ?? [];
    $item->url = $this->getItemUrl($item);
    $item->parent = $item->parent ?? NULL;
    $item->children = [];

    return $item;
  }

  /**
   * Process raw database stored menu items and renderable links to the data.
   *
   * @param object $item
   *   stdClass objects from the data storage.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The account to check URL access permission for.
   * @param \Drupal\Core\Cache\CacheableMetadata $cache
   *   The metadata cache object ot update with caching information processed
   *   from this menu item.
   *
   * @return object
   *   A processed menu item, with the renderable link information applied.
   */
  protected function processItem(object $item, AccountInterface $account, CacheableMetadata $cache): ?object {
    $this->loadItem($item);

    $processed = new \stdClass();
    $processed->id = $item->id;
    $processed->title = $item->title;

    // If this menu item is linking an associated entity, try to load the
    // entity and make the link item adjustments as needed if successful.
    if (!empty($item->entity) && $item->entity instanceof EntityInterface) {
      if (empty($processed->title)) {
        $processed->title = $item->entity->label();
      }

      $cache->addCacheableDependency($item->entity);
    }

    // Only return a processed link if a proper URL could be generated.
    if ($item->url) {
      $processed->parent = $item->parent;
      $processed->expanded = $item->expanded;
      $processed->weight = $item->weight;
      $processed->url = $item->url;
      $url = $processed->url;

      if ($url->isRouted()) {
        $access = $this->accessManager
          ->checkNamedRoute($url->getRouteName(), $url->getRouteParameters(), $account, TRUE);

        // Capture the menu link access, and any caching data that goes with
        // that access resolution. We still need this menu item even if it
        // is not accessible, for menu tree building.
        $processed->access = $access->isAllowed();
        $cache->addCacheableDependency($access);
      }
      else {
        $processed->access = TRUE;
      }

      return $processed;
    }

    // No URL could be generated, and therefore this menu item is invalid.
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getMenuItems(AssociationInterface $association, $flatten = FALSE): array {
    $menuItems = [];

    try {
      $query = $this->db
        ->select(static::TABLE_NAME, 'nav')
        ->fields('nav')
        ->condition('association', $association->id())
        ->orderBy('parent', 'ASC')
        ->orderBy('weight', 'ASC');

      foreach ($query->execute() as $item) {
        $menuItems[$item->id] = $this->loadItem($item);
      }
    }
    catch (DatabaseException $e) {
      // Issue with fetching items from the database.
    }

    // If $flatten is FALSE, build the menu into a tree structure.
    if (!$flatten) {
      $tree = [];

      foreach ($menuItems as $id => $item) {
        $parent = $item->parent ?? '';
        if (!empty($menuItems[$parent])) {
          $menuItems[$parent]->children[$id] = $item;
        }
        else {
          $tree[$id] = $item;
        }
      }
      return $tree;
    }

    return $menuItems;
  }

  /**
   * {@inheritdoc}
   */
  public function updateMenuTree(AssociationInterface $association, array $menu_items): void {
    $assocId = $association->id();

    // Only allows the update of the menu tree related data. This prevents
    // accidental data from altering the URL or URL options unintentionally.
    // Use static::saveMenuItem() to save link and attributes.
    $allowedFields = [
      'parent' => NULL,
      'enabled' => TRUE,
      'expanded' => TRUE,
      'weight' => 0,
    ];

    foreach ($menu_items as $item) {
      $fields = array_intersect_key($item, $allowedFields);

      $this->db
        ->update(static::TABLE_NAME)
        ->fields($fields)
        ->condition('id', $item['id'])
        ->condition('association', $assocId)
        ->execute();
    }

    $this->clearCache($assocId);
    Cache::invalidateTags($this->getCacheTags($association));
  }

  /**
   * {@inheritdoc}
   */
  public function getMenuItem(AssociationInterface $association, $menu_item_id): object {
    $item = $this->db
      ->select(static::TABLE_NAME, 'nav')
      ->fields('nav')
      ->condition('id', $menu_item_id)
      ->condition('association', $association->id())
      ->execute()
      ->fetchObject();

    if ($item) {
      return $this->loadItem($item);
    }

    // Report that the requested menu item isn't available.
    $msg = sprintf('Unable to find menu item ID: %s for Association ID: %s', $menu_item_id, $association->id());
    throw new \InvalidArgumentException($msg);
  }

  /**
   * {@inheritdoc}
   */
  public function saveMenuItem(AssociationInterface $association, array $item): void {
    $assocId = $association->id();
    $fields = [
      'title' => !empty($item['title']) ? serialize($item['title']) : NULL,
      'options' => serialize($item['options'] ?? []),
    ];

    if (isset($item['enabled'])) {
      $fields['enabled'] = (bool) $item['enabled'];
    }

    if (isset($item['expanded'])) {
      $fields['expanded'] = (bool) $item['expanded'];
    }

    if (empty($item['entity'])) {
      if (!empty($item['route']['route_name'])) {
        $item['route'] += ['route_parameters' => []];
        $fields['route'] = serialize($item['route']);
        $fields['uri'] = NULL;
      }
      elseif (!empty($item['uri'])) {
        $fields['uri'] = (string) $item['uri'];
        $fields['route'] = NULL;
      }
      else {
        // If not valid route or URI data is available, then this menu link
        // is invalid. A valid route "<nolink>" should be used if the menu item
        // intentionally has no URL link value.
        throw new \InvalidArgumentException('Menu item is missing URL data and cannot be saved.');
      }
    }

    if (empty($item['id'])) {
      $fields['association'] = $assocId;
      $this->db
        ->insert(static::TABLE_NAME)
        ->fields($fields)
        ->execute();
    }
    else {
      $this->db
        ->update(static::TABLE_NAME)
        ->fields($fields)
        ->condition('association', $assocId)
        ->condition('id', $item['id'])
        ->execute();
    }

    $this->clearCache($assocId);
    Cache::invalidateTags($this->getCacheTags($association));
  }

  /**
   * {@inheritdoc}
   */
  public function deleteMenuItem(AssociationInterface $association, $menu_item_id): bool {
    $assocId = $association->id();

    // Only delete if the association owns the menu item requested, and the
    // link does not have entity data (is associated entity link).
    $successful = (bool) $this->db
      ->delete(static::TABLE_NAME)
      ->condition('association', $assocId)
      ->condition('id', $menu_item_id)
      ->isNull('entity')
      ->execute();

    if ($successful) {
      $this->clearCache($assocId);
      Cache::invalidateTags($this->getCacheTags($association));
    }
    return $successful;
  }

  /**
   * {@inheritdoc}
   */
  public function getMenu(AssociationInterface $association, AccountInterface $account = NULL): array {
    $assocId = $association->id();
    $account = $account ?? $this->account;

    // Fetch and build menu items if they have not already been retrieved
    // recently for the requested user.
    if (!isset($this->menus[$assocId])) {
      $cache = new CacheableMetadata();
      $cache->addCacheTags($this->getCacheTags($association));
      $cache->addCacheContexts(['route.association']);

      $menuItems = [];
      try {
        $query = $this->db
          ->select(static::TABLE_NAME, 'nav')
          ->fields('nav')
          ->condition('association', $assocId)
          ->condition('enabled', static::ITEM_ENABLED)
          ->orderBy('parent', 'ASC')
          ->orderBy('weight', 'ASC');

        foreach ($query->execute() as $data) {
          if ($processed = $this->processItem($data, $account, $cache)) {
            $menuItems[$processed->id] = $processed;
          }
        }
      }
      catch (DatabaseException $e) {
        // Issue with fetching items from the database.
      }

      $this->menus[$assocId] = [
        'cache' => $cache,
        'id' => $assocId,
        'items' => $menuItems,
      ];
    }

    return $this->menus[$assocId];
  }

  /**
   * {@inheritdoc}
   */
  public function deleteMenu(AssociationInterface $association): void {
    $assocId = $association->id();

    $this->db
      ->delete(static::TABLE_NAME)
      ->condition('association', $assocId)
      ->execute();

    $this->clearCache($assocId);
    Cache::invalidateTags($this->getCacheTags($association));
  }

  /**
   * {@inheritdoc}
   */
  public function addAssociated(AssociatedEntityInterface $entity, array $options = []): void {
    if ($association = $entity->getAssociation()) {
      $assocId = $association->id();
      $type = $entity->getEntityTypeId();

      // Only insert if not already included as a menu item.
      $this->db
        ->merge(static::TABLE_NAME)
        ->keys([
          'association' => $assocId,
          'entity' => $type . ':' . $entity->id(),
        ])
        ->insertFields([
          'association' => $assocId,
          'entity' => $type . ':' . $entity->id(),
          'enabled' => static::ITEM_ENABLED,
          'options' => serialize($options),
        ])
        ->execute();

      $this->clearCache($assocId);
      Cache::invalidateTags($this->getCacheTags($association));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function removeAssociated(AssociatedEntityInterface $entity): void {
    $this->db
      ->delete(static::TABLE_NAME)
      ->condition('entity', $entity->getEntityTypeId() . ':' . $entity->id())
      ->execute();

    if ($association = $entity->getAssociation()) {
      $this->clearCache($association->id());
      Cache::invalidateTags($this->getCacheTags($association));
    }
  }

}
