<?php

namespace Drupal\association\Adapter;

use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Access\AccessResultInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\association\Event\AssociationEvents;
use Drupal\association\Event\AccessConditionAlterEvent;

/**
 * The default EntityAdapter implementation.
 *
 * This adapter employs the common entity storage creation and form operation
 * fetching, and will work for most standard entity types.
 *
 * This adapter also has plugin definition settings for limiting allowed bundles
 * and some different handling of entity administrative permissions.
 *
 * Potential reasons to create custom adapters:
 *   - Entity requires additional values during creation
 *   - Entity form requires additional settings, or initialization
 *   - Entity has non-standard operations or special access logic
 *
 * Definitions can be overridden or altered by implementing the
 * hook_association_entity_adapter_info_alter() alter hook.
 */
class EntityAdapter extends PluginBase implements EntityAdapterInterface, ContainerFactoryPluginInterface {

  use AssociatedQueryAlterTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The entity type bundle info manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $entityBundleInfo;

  /**
   * The form builder.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The event dispatcher.
   *
   * @var \Symfony\Component\EventDispatcher\EventDispatcherInterface
   */
  protected $eventDispatcher;

  /**
   * List of entity bundles supported for association link target entities.
   *
   * @var \Drupal\Core\StringTranslation\TranslatableMarkup[]|string[]
   */
  protected $bundles;

  /**
   * Create a new instance of EntityAdapter class.
   *
   * @param array $configuration
   *   Configurations for the adapter plugin.
   * @param string $plugin_id
   *   The unique plugin ID.
   * @param mixed $plugin_definition
   *   Plugin definitions from discovery.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle info manager.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder.
   * @param Symfony\Component\EventDispatcher\EventDispatcherInterface $event_dispatcher
   *   The event dispatcher.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, EntityTypeBundleInfoInterface $entity_type_bundle_info, FormBuilderInterface $form_builder, EventDispatcherInterface $event_dispatcher) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->entityTypeManager = $entity_type_manager;
    $this->entityBundleInfo = $entity_type_bundle_info;
    $this->formBuilder = $form_builder;
    $this->eventDispatcher = $event_dispatcher;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('entity_type.bundle.info'),
      $container->get('form_builder'),
      $container->get('event_dispatcher')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityTypeId() {
    return $this->pluginDefinition['entity_type'];
  }

  /**
   * {@inheritdoc}
   */
  public function getLabel() {
    if (!empty($this->pluginDefinition['label'])) {
      return $this->pluginDefinition['label'];
    }

    $entityType = $this->entityTypeManager
      ->getDefinition($this->getEntityTypeId(), FALSE);

    return $entityType ? $entityType->getLabel() : $this->t('Unknown');
  }

  /**
   * {@inheritdoc}
   */
  public function getBundles() {
    if (!isset($this->bundles)) {
      $entityTypeId = $this->getEntityTypeId();
      $bundleInfo = $this->entityBundleInfo
        ->getBundleInfo($entityTypeId);

      $this->bundles = [];
      foreach ($bundleInfo as $name => $info) {
        $this->bundles[$name] = $info['label'];
      }
    }

    return $this->bundles;
  }

  /**
   * {@inheritdoc}
   */
  public function getOperations(ContentEntityInterface $entity) {
    $entityTypeId = $this->getEntityTypeId();

    if ($entityTypeId !== $entity->getEntityTypeId()) {
      $msg = sprintf('Entity is of wrong type, expected: %s but got: %s', $entityTypeId, $entity->getEntityTypeId());
      throw new \InvalidArgumentException($msg);
    }

    return $this->entityTypeManager
      ->getListBuilder($entityTypeId)
      ->getOperations($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function checkAccess(ContentEntityInterface $entity, $op, AccountInterface $account): AccessResultInterface {
    if ($entity->getEntityTypeId() != $this->getEntityTypeId()) {
      throw new \InvalidArgumentException(sprintf('Entity adapter expected %s but got %s.', $this->getEntityTypeId(), $entity->getEntityTypeId()));
    }

    if ($entity->hasField('associations') && ($assocLink = $entity->associations->entity)) {
      $assoc = $assocLink->getAssociation();

      switch ($op) {
        case 'view':
          $viewAccess = $assoc->access('view', $account, TRUE);
          if ($viewAccess->isAllowed()) {
            // Access will be up to the entity's own visibility rules from here.
            // View access is only blocked here if the association is inactive,
            // but otherwise the entity is responsible for further access or
            // publish state checking.
            return AccessResult::neutral()->addCacheableDependency($viewAccess);
          }

          // Purposely fall-through and allow view if user has access to edit
          // or manage this content directly.
        case 'edit':
        case 'update':
          $access = $assoc->access('manage', $account, TRUE);
          break;

        case 'delete':
          $access = $assoc->access('delete_content', $account, TRUE);
          break;

        default:
          // Don't exert any control over any other operations.
          return AccessResult::neutral()->addCacheableDependency($assoc);
      }

      // If edit, update or delete were not explicitly granted, we want to
      // deny access to these operations. It is possible to alter access grants
      // at the association access layer.
      if (!$access->isAllowed()) {
        // Ensure that access is forbidden but make sure not to lose the
        // caching metadata accumulated up to this point.
        $access = AccessResult::forbidden()->addCacheableDependency($access);
      }

      return $access->addCacheableDependency($assoc);
    }

    return AccessResult::neutral();
  }

  /**
   * {@inheritdoc}
   */
  public function accessQueryAlter(SelectInterface $query, $op, AccountInterface $account): void {
    $aliases = $this->ensureTables($query);
    $accessCond = $this->buildAccessCondition($query, $aliases, $op, $account);

    // Give other modules a chance to alter the query or the access conditions.
    $event = new AccessConditionAlterEvent($op, $query, $accessCond, $aliases, $account);
    $this->eventDispatcher->dispatch($event, AssociationEvents::ACCESS_CONDITION_ALTER);

    if ($accessCond) {
      // Top level conjunction is always "AND" so it can just be added to
      // the query as just another conditional. If this ends up being an "OR"
      // query, this assumption will need to be revisited.
      $query->condition($accessCond);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function createEntity($bundle, $values = []): ?ContentEntityInterface {
    try {
      $entityTypeId = $this->getEntityTypeId();
      $entityType = $this->entityTypeManager
        ->getDefinition($entityTypeId);

      if ($bundleKey = $entityType->getKey('bundle')) {
        $values[$bundleKey] = $bundle instanceof EntityInterface ? $bundle->id() : $bundle;
      }

      return $this->entityTypeManager
        ->getStorage($entityTypeId)
        ->create($values);
    }
    catch (PluginException $e) {
      return NULL;
    }
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityForm(ContentEntityInterface $entity, $op, FormStateInterface &$form_state) {
    $entityType = $entity->getEntityType();

    if ($entityType->id() != $this->getEntityTypeId()) {
      throw new \InvalidArgumentException('This entity type adapter only supports entities of type ' . $this->getEntityTypeId());
    }

    // Allow the operation value to fallback to a "default" form if needed,
    // with the exception of the "delete" operation which should not fallback.
    $operation = ($op == 'delete' || $entityType->getFormClass($op)) ? $op : 'default';

    // Create the form object of the appropriate type, and build the form.
    // Using the form builder instead of the entity form builder, so we are able
    // to manage the form state which is used for the form building.
    $formObj = $this->entityTypeManager->getFormObject($entityType->id(), $operation);
    $formObj->setEntity($entity);
    $formObj->setOperation($op);

    return $this->formBuilder->buildForm($formObj, $form_state);
  }

}
