<?php

namespace Drupal\association\Behavior\Manager;

use Drupal\Core\Url;
use Drupal\association\Plugin\BehaviorInterface;
use Drupal\association\Behavior\ManagerBuilderBase;

/**
 * Build the content management controls for an Entity Manifest behavior plugin.
 */
class EntityManifestBuilder extends ManagerBuilderBase {

  /**
   * {@inheritdoc}
   */
  public function buildManagerUi() {
    $plugin = $this->association->getBehavior();
    $manifest = $plugin->getConfiguration()['manifest'] ?? [];
    $canCreate = $this->association->access('create_content');

    $byTags = [];
    foreach ($this->getLinkedEntities() as $entity) {
      $byTags[$entity->tag->value][$entity->id()] = $entity;
    }

    $elements = [];

    // Iterate through the manifest types and create a section for each.
    foreach ($manifest as $tag => $def) {
      $elements[$tag] = [
        '#type' => 'container',
        '#attributes' => [
          'class' => ['manifest-section'],
        ],

        'title' => [
          '#type' => 'html_tag',
          '#tag' => 'h3',
          '#attributes' => [
            'class' => ['manifest-section__title'],
          ],
          '#value' => $def['label'],
        ],
      ];

      if (!empty($byTags[$tag])) {
        $table = [
          '#type' => 'table',
          '#header' => [
            'title' => $this->t('Title'),
            'status' => $this->t('Status'),
            'op' => $this->t('Operations'),
          ],
        ];

        foreach ($byTags[$tag] as $entityId => $entity) {
          $table[$entityId] = [
            'title' => $this->getEntityLink($entity),
            'status' => [
              '#markup' => $entity->getStatus() ? $this->t('Published') : $this->t('Unpublished'),
            ],
            'op' => [
              '#type' => 'operations',
              '#links' => $this->getEntityOperations($entity),
            ],
          ];
        }

        $elements[$tag]['table'] = $table;
      }

      // User has permission to create content, and the limit for # of items
      // has not been reached.
      if ($canCreate && ($def['limit'] == BehaviorInterface::CARDINALITY_UNLIMITED || empty($byTags[$tag]) || count($byTags[$tag]) < $def['limit'])) {
        $elements[$tag]['add_more'] = [
          '#type' => 'link',
          '#title' => $this->t('Add @label', [
            '@label' => $def['label'],
          ]),
          '#attributes' => [
            'class' => [
              'button',
              'button--primary',
            ],
          ],
          '#url' => Url::fromRoute('entity.association_link.add_content', [
            'association' => $this->association->id(),
            'tag' => $tag,
          ]),
        ];
      }
    }

    return $elements;
  }

}
