<?php

namespace Drupal\association;

use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Component\Plugin\Exception\InvalidPluginDefinitionException;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\toolshed\Plugin\StaticYamlPluginManager;
use Drupal\association\Adapter\EntityAdapterInterface;

/**
 * The entity type adapter manager for Entity Association linked content.
 */
class EntityAdapterManager extends StaticYamlPluginManager implements EntityAdapterManagerInterface {

  /**
   * Maps Entity type IDs (keys) to adapter plugin IDs (values).
   *
   * @var string[]
   */
  protected $entityTypes;

  /**
   * Get the mapping from entity type ID to an adapter plugin ID.
   *
   * @return array
   *   Array that works as a lookup for entity types the appropriate adapter
   *   plugin to use. The array keys are the entity type IDs and the values are
   *   the adapter plugin IDs.
   */
  protected function getEntityTypeMap(): array {
    if (!isset($this->entityTypes)) {
      $this->entityTypes = [];

      foreach ($this->getDefinitions() as $pluginId => $definition) {
        $this->entityTypes[$definition['entity_type']] = $pluginId;
      }
    }

    return $this->entityTypes;
  }

  /**
   * {@inheritdoc}
   */
  public function isAssociableType($entity_type_id): bool {
    $allowedTypes = $this->getEntityTypeMap();
    return isset($allowedTypes[$entity_type_id]);
  }

  /**
   * {@inheritdoc}
   */
  public function isAssociable(EntityInterface $entity): bool {
    return $this->isAssociableType($entity->getEntityTypeId())
      && $entity instanceof ContentEntityInterface
      && $entity->hasField('associations');
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityTypes(): array {
    $typeIds = array_keys($this->getEntityTypeMap());
    return array_combine($typeIds, $typeIds);
  }

  /**
   * {@inheritdoc}
   */
  public function getAdapter(EntityInterface $entity, $exception_on_invalid = TRUE): ?EntityAdapterInterface {
    return $this->getAdapterByEntityType($entity->getEntityTypeId(), $exception_on_invalid);
  }

  /**
   * {@inheritdoc}
   */
  public function getAdapterByEntityType($entity_type_id, $exception_on_invalid = TRUE): ?EntityAdapterInterface {
    $allowedTypes = $this->getEntityTypeMap();
    $pluginId = $allowedTypes[$entity_type_id] ?? NULL;

    if ($pluginId) {
      return $this->createInstance($pluginId);
    }

    if ($exception_on_invalid) {
      // Report that an adapter for this entity type is not available.
      throw new PluginNotFoundException($entity_type_id, "No entity type adapter defined for %s was found.");
    }
    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  protected function findDefinitions() {
    $definitions = $this->getDiscovery()->getDefinitions();

    foreach ($definitions as $plugin_id => &$definition) {
      $this->processDefinition($definition, $plugin_id);
    }

    if ($this->extensionHandler instanceof ModuleHandlerInterface) {
      $this->extensionHandler->alter('association_entity_adapter_info', $definitions);
    }

    return $definitions;
  }

  /**
   * {@inheritdoc}
   */
  protected function processDefinition(&$definition, $plugin_id) {
    if (!is_array($definition)) {
      return;
    }

    if (empty($definition['entity_type'])) {
      throw new InvalidPluginDefinitionException($plugin_id, "Entity adapter plugin definition '{$plugin_id}' does not contain an entity_type.");
    }

    if (empty($definition['class'])) {
      $definition['class'] = $this->defaultPluginClass;
    }
  }

}
