<?php

namespace Drupal\association\Plugin\Association\Behavior;

use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Component\Plugin\Exception\PluginException;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Plugin\PluginWithFormsInterface;
use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Plugin\PluginWithFormsTrait;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\association\EntityAdapterManagerInterface;
use Drupal\association\Plugin\BehaviorInterface;
use Drupal\association\Entity\AssociationInterface;
use Drupal\association\Entity\AssociationTypeInterface;

/**
 * Behavior for managing associations which have a prescriptive entity manifest.
 *
 * Manages Entity Associations that need to maintain the specific counts of
 * entities and bundles allowed. Each entity bundle can have a separate
 * cardinality constraint. This gives a associations a more templated content
 * building experience.
 *
 * @AssociationBehavior(
 *   id = "entity_manifest",
 *   label = @Translation("Entity Manifest"),
 *   manager_builder = "\Drupal\association\Behavior\Manager\EntityManifestBuilder",
 *   forms = {
 *     "configure" = "\Drupal\association\Behavior\Form\ConfigureManifestBehaviorForm",
 *   }
 * )
 */
class EntityManifestBehavior extends PluginBase implements BehaviorInterface, PluginWithFormsInterface, ContainerFactoryPluginInterface {

  use PluginWithFormsTrait;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Manager entity type adapters for associating to association entities.
   *
   * @var \Drupal\association\EntityAdapterManagerInterface
   */
  protected $adapterManager;

  /**
   * Create a new instance of the ContentManifestBehavior plugin.
   *
   * @param array $configuration
   *   Plugin configuration options.
   * @param string $plugin_id
   *   The plugin identifier.
   * @param mixed $plugin_definition
   *   The plugin definition (from plugin discovery).
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\association\EntityAdapterManagerInterface $entity_adapter_manager
   *   Manager entity type adapters for associating to association entities.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager, EntityAdapterManagerInterface $entity_adapter_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->entityTypeManager = $entity_type_manager;
    $this->adapterManager = $entity_adapter_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('plugin.manager.association.entity_adapter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defaultConfiguration() {
    return [
      'manifest' => [],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getConfiguration() {
    return $this->configuration;
  }

  /**
   * {@inheritdoc}
   */
  public function setConfiguration(array $configuration) {
    $this->configuration = $configuration + $this->defaultConfiguration();
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityTypes() {
    $entityTypes = [];
    $manifest = $this->getConfiguration()['manifest'] ?? [];
    $allowedTypes = $this->adapterManager->getEntityTypes();

    foreach ($manifest as $tagInfo) {
      [$type] = explode(':', $tagInfo['entity_bundle']);

      if ($type && isset($allowedTypes[$type])) {
        $entityTypes[$type] = $type;
      }
    }

    return $entityTypes;
  }

  /**
   * {@inheritdoc}
   */
  public function getTagLabel($tag) {
    return $this->configuration['manifest'][$tag]['label'] ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getManagerBuilderClass() {
    return $this->pluginDefinition['manager_builder'];
  }

  /**
   * {@inheritdoc}
   */
  public function createAccess(AssociationInterface $assoc, $tag, AccountInterface $account) {
    if (!empty($this->configuration['manifest'][$tag])) {
      $definition = $this->configuration['manifest'][$tag];

      // If not an unlimited number of items, we need to ensure that it's okay
      // to create another new instance of the entity type.
      if ($definition['limit'] !== BehaviorInterface::CARDINALITY_UNLIMITED) {
        $query = $this->entityTypeManager->getStorage('association_link')->getQuery();
        $count = $query->condition('association', $assoc->id())
          ->condition('tag', $tag)
          ->count()
          ->execute();

        return $count < $definition['limit'] ? AccessResult::allowed() : AccessResult::forbidden();
      }
      else {
        return AccessResult::allowed();
      }
    }

    return AccessResult::forbidden();
  }

  /**
   * {@inheritdoc}
   */
  public function createTagEntity(AssociationInterface $association, $tag) {
    if (!empty($this->configuration['manifest'][$tag]['entity_bundle'])) {
      [$entityTypeId, $bundle] = explode(':', $this->configuration['manifest'][$tag]['entity_bundle']);

      try {
        return $this->adapterManager
          ->getAdapterByEntityType($entityTypeId)
          ->createEntity($bundle);
      }
      catch (PluginException $e) {
        // Unable to fetch the entity adapter for this entity type and therefore
        // we should fail to create the entity.
      }
    }

    return NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigUpdate(AssociationTypeInterface $assocation_type, array $changes) {
    $errors = [];
    $current = $this->getConfiguration()['manifest'] ?? [];
    $missing = array_diff_key($current, $changes['manifest']);

    if ($missing) {
      $errors[] = $this->t('Entities have been removed from the manifest after association type has data: @types', [
        '@types' => implode(', ', array_keys($missing)),
      ]);
    }
    else {
      foreach ($current as $tag => $info) {
        $tagChange = $changes['manifest'][$tag];

        if ($tagChange['required'] != $info['required'] || $tagChange['entity_bundle'] != $info['entity_bundle']) {
          $errors[] = $this->t('Entity manifest bundle and required properties cannot be changed for @tag', [
            '@tag' => $tag,
          ]);
        }

        if ($tagChange['limit'] < $info['limit']) {
          $errors[] = $this->t('Entity manifest limit properties cannot be changed for @tag', [
            '@tag' => $tag,
          ]);
        }
      }
    }

    return $errors;
  }

}
