<?php

namespace Drupal\association\Entity;

use Drupal\association\Plugin\BehaviorInterface;
use Drupal\Core\Entity\ContentEntityInterface;
use Drupal\Core\Entity\EntityChangedInterface;
use Drupal\Core\Entity\EntityPublishedInterface;
use Drupal\user\EntityOwnerInterface;

/**
 * Base interface for the association entity.
 */
interface AssociationInterface extends ContentEntityInterface, EntityChangedInterface, EntityOwnerInterface, EntityPublishedInterface {

  /**
   * Gets the purging state of this entity association. True during deletion.
   *
   * @return bool
   *   The state of the association's purging state. TRUE if the association
   *   is in the process of deleting associated content, and other associated
   *   data.
   */
  public function isPurging();

  /**
   * Gets the association type instance for this association.
   *
   * @return \Drupal\association\Entity\AssociationType
   *   Get the bundle entity for this association.
   */
  public function getType();

  /**
   * Get the behavior plugin to use with this association entity.
   *
   * @return \Drupal\association\Plugin\BehaviorInterface|null
   *   Returns the association type behavior plugin that should be used
   *   with this association entity.
   */
  public function getBehavior(): ?BehaviorInterface;

  /**
   * Add an entity to the association.
   *
   * @param string $tag
   *   The association behavior tag.
   * @param \Drupal\Core\Entity\ContentEntityInterface $entity
   *   The entity to associate. The "associations" field will get updated
   *   with the association_link.
   * @param bool $save_link
   *   (optional) Should the association link be saved? Can only happen if
   *   $entity has been saved and has a valid entity ID.
   *
   * @return \Drupal\association\Entity\AssociationLink
   *   The association_link instance created to link the association and entity.
   *
   * @throws \Drupal\association\Entity\Exception\AlreadyAssociatedException
   */
  public function associateEntity($tag, ContentEntityInterface $entity, $save_link = TRUE): ?AssociationLink;

  /**
   * Indicate if the association is enabled.
   *
   * @return bool
   *   Indicate if the association is currently active.
   */
  public function isActive();

  /**
   * Get the companion landing page if one is available.
   *
   * @return \Drupal\association\Entity\AssociationPage|false
   *   The companion association page if there is one, otherwise boolean FALSE.
   */
  public function getPage();

}
