<?php

namespace Drupal\association\Entity;

use Drupal\association\Plugin\BehaviorInterface;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Config\Entity\ConfigEntityInterface;
use Drupal\Core\Entity\RevisionableEntityBundleInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\toolshed\Entity\EntityBundleBase;
use Symfony\Component\DependencyInjection\Exception\ServiceNotFoundException;

/**
 * Configuration entity to be used as the bundle type for association entities.
 *
 * @ConfigEntityType(
 *   id = "association_type",
 *   label = @Translation("Association type"),
 *   label_plural = @Translation("Association types"),
 *   config_prefix = "type",
 *   admin_permission = "administer association configurations",
 *   bundle_of = "association",
 *   entity_keys = {
 *     "id" = "id",
 *     "label" = "label",
 *     "uuid" = "uuid",
 *   },
 *   handlers = {
 *     "list_builder" = "Drupal\association\Entity\Controller\AssociationTypeListBuilder",
 *     "form" = {
 *       "default" = "Drupal\association\Entity\Form\AssociationTypeForm",
 *       "edit" = "Drupal\association\Entity\Form\AssociationTypeForm",
 *       "delete" = "Drupal\toolshed\Entity\Form\EntityBundleDeleteConfirm",
 *     },
 *     "route_provider" = {
 *       "html" = "Drupal\Core\Entity\Routing\AdminHtmlRouteProvider",
 *     },
 *   },
 *   links = {
 *     "collection" = "/admin/structure/association",
 *     "add-form" = "/admin/structure/association/add",
 *     "edit-form" = "/admin/structure/association/manage/{association_type}",
 *     "delete-form" = "/admin/structure/association/manage/{association_type}/delete",
 *   },
 *   config_export = {
 *     "id",
 *     "label",
 *     "hasPage",
 *     "searchable",
 *     "newRevision",
 *     "behavior",
 *   },
 * )
 */
class AssociationType extends EntityBundleBase implements AssociationTypeInterface, ConfigEntityInterface, RevisionableEntityBundleInterface {

  use StringTranslationTrait;

  /**
   * The entity ID.
   *
   * @var string
   */
  protected $id;

  /**
   * The human friendly display name for the association type.
   *
   * @var string
   */
  protected $label;

  /**
   * Does this association type have a default view and landing page.
   *
   * @var bool
   */
  protected $hasPage;

  /**
   * Indicates if content for associations of this type should be searchable.
   *
   * @var bool
   */
  protected $searchable;

  /**
   * Does this association type already have data?
   *
   * @var bool
   */
  protected $hasData;

  /**
   * Should the new association pages create new revisions by default.
   *
   * @var bool
   */
  protected $newRevision = TRUE;

  /**
   * The behavior plugin definition (ID and configurations).
   *
   * @var array
   */
  protected $behavior;

  /**
   * The loaded association behavior handling plugin.
   *
   * @var \Drupal\association\Plugin\BehaviorInterface
   */
  protected $loadedBehavior;

  /**
   * Get the plugin manager for creating new plugin instances.
   *
   * @return \Drupal\association\Plugin\BehaviorPluginManagerInterface
   *   Plugin manager for creating and managing association behavior plugins.
   */
  protected static function getBehaviorManager() {
    return \Drupal::service('plugin.manager.association.behavior');
  }

  /**
   * {@inheritdoc}
   */
  public function set($property_name, $value) {
    parent::set($property_name, $value);

    // Ensure that if the behavior configurations are updated, the loaded
    // plugin behavior will get rebuilt.
    if ($property_name === 'behavior') {
      unset($this->loadedBehavior);
    }

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function isContentSearchable(): bool {
    return $this->searchable ?? FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function hasPage() {
    return (bool) $this->hasPage;
  }

  /**
   * {@inheritdoc}
   */
  public function hasData() {
    if (!isset($this->hasData)) {
      $this->hasData = parent::hasData();
    }

    return $this->hasData;
  }

  /**
   * {@inheritdoc}
   */
  public function shouldCreateNewRevision() {
    return $this->newRevision;
  }

  /**
   * {@inheritdoc}
   */
  public function getBehavior($force_rebuild = FALSE): ?BehaviorInterface {
    if (!isset($this->loadedBehavior) || $force_rebuild) {
      try {
        if (empty($this->behavior['id'])) {
          throw new PluginNotFoundException('No handler plugin specified.');
        }

        $this->behavior += ['config' => []];
        $this->loadedBehavior = static::getBehaviorManager()
          ->createInstance($this->behavior['id'], $this->behavior['config']);
      }
      catch (ServiceNotFoundException | PluginNotFoundException $e) {
        $this->loadedBehavior = FALSE;

        // If we are creating a new bundle, this is still under construction
        // and should not be logged. If not new this could mean a missing plugin
        // or module and should be logged.
        if (!$this->isNew()) {
          watchdog_exception('association', $e);
        }
      }
    }

    return $this->loadedBehavior ?: NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function validateConfigChanges(array $updates) {
    $errors = [];

    // These configurations are only locked if association type has data.
    if ($this->hasData()) {
      if ($this->hasPage() != $updates['hasPage']) {
        $errors[] = $this->t('Association type %id cannot change its page settings after association entities have been created', [
          '%id' => $this->id(),
        ]);
      }

      // Check for plugin behavior changes.
      if ($this->behavior['id'] !== $updates['behavior']['id']) {
        $errors[] = $this->t('Association type %id cannot change its behavior plugin after association entities have been created', [
          '%id' => $this->id(),
        ]);
      }
      elseif ($behavior = $this->getBehavior()) {
        $errors = array_merge($errors, $behavior->validateConfigUpdate($this, $updates['behavior']['config']));
      }
    }

    return $errors;
  }

}
