<?php

namespace Drupal\association\Entity\Controller;

use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\Url;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * List builder for association types.
 */
class AssociationTypeListBuilder extends ConfigEntityListBuilder {

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Create a new instance of the AssociationTypeListBuilder.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type instance.
   * @param \Drupal\Core\Entity\EntityStorageInterface $entity_storage
   *   The entity storage for this entity type.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $entity_storage, AccountInterface $current_user) {
    parent::__construct($entity_type, $entity_storage);

    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('current_user')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['label'] = $this->t('Association type');
    $header['id'] = $this->t('Machine name');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {
    $row['label'] = $entity->label();
    $row['id'] = $entity->id();

    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultOperations(EntityInterface $entity) {
    $operations = parent::getDefaultOperations($entity);

    /** @var \Drupal\association\Entity\AssociationTypeInterface $entity */
    if ($entity->hasPage() && $this->moduleHandler->moduleExists('field_ui')) {
      $urlParams = ['association_type' => $entity->id()];

      if ($this->currentUser->hasPermission('administer assocation_page fields')) {
        $url = Url::fromRoute('entity.association_page.field_ui_fields', $urlParams);
        $operations['manage-page-fields'] = [
          'title' => $this->t('Page fields'),
          'weight' => 50,
          'url' => $this->ensureDestination($url),
        ];
      }

      if ($this->currentUser->hasPermission('administer assocation_page form display')) {
        $url = Url::fromRoute('entity.entity_form_display.association_page.default', $urlParams);
        $operations['manage-page-form-display'] = [
          'title' => $this->t('Page form display'),
          'weight' => 55,
          'url' => $this->ensureDestination($url),
        ];
      }

      if ($this->currentUser->hasPermission('administer assocation_page display')) {
        $url = Url::fromRoute('entity.entity_view_display.association_page.default', $urlParams);
        $operations['manage-page-form-display'] = [
          'title' => $this->t('Page display'),
          'weight' => 55,
          'url' => $this->ensureDestination($url),
        ];
      }
    }

    return $operations;
  }

}
