<?php

namespace Drupal\association\Entity\Views;

use Drupal\association\EntityAdapterManagerInterface;
use Drupal\Component\Plugin\Exception\PluginNotFoundException;
use Drupal\Core\Entity\EntityFieldManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\views\EntityViewsData;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides the views data for the association link entities.
 */
class AssociationLinkViewsData extends EntityViewsData {

  /**
   * The entity type adapter plugin manager.
   *
   * @var \Drupal\association\EntityAdapterManagerInterface
   */
  protected $entityAdapterManager;

  /**
   * Constructs a AssociationLinkViewsData object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type to provide views integration for.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\StringTranslation\TranslationInterface $translation_manager
   *   The translation manager.
   * @param \Drupal\Core\Entity\EntityFieldManagerInterface $entity_field_manager
   *   The entity field manager.
   * @param \Drupal\association\EntityAdapterManagerInterface $entity_adapter_manager
   *   Manager entity type adapters for associating to association entities.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityTypeManagerInterface $entity_type_manager, ModuleHandlerInterface $module_handler, TranslationInterface $translation_manager, EntityFieldManagerInterface $entity_field_manager, EntityAdapterManagerInterface $entity_adapter_manager) {
    parent::__construct(
      $entity_type,
      $entity_type_manager->getStorage($entity_type->id()),
      $entity_type_manager,
      $module_handler,
      $translation_manager,
      $entity_field_manager
    );

    $this->entityAdapterManager = $entity_adapter_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
      $container->get('string_translation'),
      $container->get('entity_field.manager'),
      $container->get('plugin.manager.association.entity_adapter')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getViewsData() {
    $data = parent::getViewsData();

    // The association link entity table to base fields and requests from.
    $baseTable = $this->entityType->getDataTable() ?: $this->entityType->getBaseTable();

    // Add a custom numeric argument for the parent association ID that allows
    // us to use replacement titles with the parent's label.
    $data[$baseTable]['association']['argument'] = [
      'id' => 'association_id',
      'numeric' => TRUE,
    ];

    // Unset the 'target' field relationship as this is dynamic based on the
    // bundle type of the association.
    unset($data[$baseTable]['target']['relationship']);

    foreach ($this->entityAdapterManager->getEntityTypes() as $entityType) {
      try {
        $entityDef = $this->entityTypeManager->getDefinition($entityType);
        $entityLabel = $entityDef->getLabel();
        $entityTable = $entityDef->getDataTable() ?: $entityDef->getBaseTable();

        $data[$entityTable]['association_link'] = [
          'title' => $this->entityType->getLabel(),
          'help' => $this->t('The @entity_type linked with a association entity.', [
            '@entity_type' => $entityLabel,
          ]),
          'relationship' => [
            'id' => 'standard',
            'group' => $this->entityType->getLabel(),
            'base' => $baseTable,
            'base field' => 'target',
            'relationship field' => $entityDef->getKey('id'),
            'extra' => [
              [
                'field' => 'entity_type',
                'value' => $entityDef->id(),
              ],
            ],
            'target_entity_type' => $this->entityType->id(),
          ],
        ];
      }
      catch (PluginNotFoundException $e) {
        // Entity type is missing, probably the module has been uninstalled and
        // the entity settings need to get updated. Skip the type without
        // breaking completely.
      }
    }

    return $data;
  }

}
